﻿unit Artifacts_Browsers;

interface

uses
{$IF DEFINED (ISFEXGUI)}
  GUI, Modules,
{$IFEND}
  ESE,
  ByteStream, Classes, Clipbrd, Common, Contnrs, DataEntry, DataStorage, DateUtils, DIRegex,
  Graphics, JSON, PropertyList, Math, ProtoBuf, Regex, SQLite3, SysUtils, Variants, XML,
  Artifact_Utils      in 'Common\Artifact_Utils.pas',
  Columns             in 'Common\Column_Arrays\Columns.pas',
  Icon_List           in 'Common\Icon_List.pas',
  Itunes_Backup_Sig   in 'Common\Itunes_Backup_Signature_Analysis.pas',
  RS_DoNotTranslate   in 'Common\RS_DoNotTranslate.pas',
  Browser_Columns     in 'Common\Column_Arrays\Arrays_Browser.pas';

type
  TDataStoreFieldArray = array of TDataStoreField;

  TSQL_FileSearch = record
    fi_Carve_Adjustment           : integer;
    fi_Carve_Footer               : string;
    fi_Carve_Header               : string;
    fi_ESE_ContainerName          : string;
    fi_Glob1_Search               : string;
    fi_Glob2_Search               : string;
    fi_Glob3_Search               : string;
    fi_Icon_Category              : integer;
    fi_Icon_OS                    : integer;
    fi_Icon_Program               : integer;
    fi_Name_OS                    : string;
    fi_Name_Program               : string;
    fi_Name_Program_Type          : string;
    fi_NodeByName                 : string;
    fi_Process_As                 : string;
    fi_Process_ID                 : string;
    fi_Recovery_Type              : string;
    fi_Reference_Info             : string;
    fi_Regex_Search               : string;
    fi_Rgx_Itun_Bkup_Dmn          : string;
    fi_Rgx_Itun_Bkup_Nme          : string;
    fi_RootNodeName               : string;
    fi_Signature_Parent           : string;
    fi_Signature_Sub              : string;
    fi_SQLPrimary_Tablestr        : string;
    fi_SQLStatement               : string;
    fi_SQLTables_Required         : string;
    fi_Test_Data                  : string;
  end;

  PSQL_FileSearch = ^TSQL_FileSearch;
  TgArr = array of TSQL_FileSearch;

const
  ARTIFACTS_DB              = 'Artifacts_Browsers.db';
  CATEGORY_NAME             = 'Browsers';
  PROGRAM_NAME              = 'Browsers';
  SCRIPT_DESCRIPTION        = 'Extract Browsers Artifacts';
  SCRIPT_NAME               = 'Browsers.pas';
  // ~~~
  ATRY_EXCEPT_STR           = 'TryExcept: '; // noslz
  ANDROID                   = 'Android'; // noslz
  BL_BOOKMARK_SOURCE        = False;
  BL_PROCEED_LOGGING        = False;
  BL_USE_FLAGS              = False;
  BL_BIOME_LOGGING          = False;
  BS                        = '\';
  CANCELED_BY_USER          = 'Canceled by user.';
  CHAR_LENGTH               = 80;
  COLON                     = ':';
  COMMA                     = ',';
  CR                        = #13#10;
  DCR                       = #13#10 + #13#10;
  DNT_AccessCount           = 'DNT_AccessCount';
  DNT_AccessedTime          = 'DNT_AccessedTime';
  DNT_ContainerID           = 'DNT_ContainerId';
  DNT_CreationTime          = 'DNT_CreationTime';
  DNT_EntryID               = 'DNT_EntryId';
  DNT_ExpiryTime            = 'DNT_ExpiryTime';
  DNT_Filename              = 'DNT_Filename';
  DNT_FileOffset            = 'DNT_FileOffset';
  DNT_Filesize              = 'DNT_FileSize';
  DNT_Flags                 = 'DNT_Flags';
  DNT_Hostname              = 'DNT_HostName';
  DNT_ModifiedTime          = 'DNT_ModifiedTime';
  DNT_PageTitle             = 'DNT_PageTitle';
  DNT_Port                  = 'DNT_Port';
  DNT_RecoveryType          = 'DNT_RecoveryType';
  DNT_Reference             = 'DNT_Reference';
  DNT_ResponseHeaders       = 'DNT_ResponseHeaders';
  DNT_SyncTime              = 'DNT_SyncTime';
  DNT_Url                   = 'DNT_Url';
  DNT_UrlHash               = 'DNT_UrlHash';
  DNT_UrlSchemaType         = 'DNT_UrlSchemaType';
  DNT_WebCacheFileName      = '^WebCacheV01\.dat';
  DUP_FILE                  = '( \d*)?$';
  FBN_ITUNES_BACKUP_DOMAIN  = 'iTunes Backup Domain'; // noslz
  FBN_ITUNES_BACKUP_NAME    = 'iTunes Backup Name'; // noslz
  GFORMAT_STR               = '%-1s %-12s %-8s %-15s %-25s %-30s %-20s';
  GROUP_NAME                = 'Artifact Analysis';
  HYPHEN                    = ' - ';
  HYPHEN_NS                 = '-';
  ICON_ANDROID              = 1017;
  ICON_IOS                  = 1062;
  IOS                       = 'iOS'; // noslz
  MAX_CARVE_SIZE            = 1024 * 10;
  MIN_CARVE_SIZE            = 0;
  PIPE                      = '|'; // noslz
  PROCESS_AS_CARVE          = 'PROCESSASCARVE'; // noslz
  PROCESS_AS_ESE_CARVE_CONTENT              = 'PROCESS_AS_ESE_CARVE_CONTENT';
  PROCESS_AS_ESE_CARVE_COOKIES              = 'PROCESS_AS_ESE_CARVE_COOKIES';
  PROCESS_AS_ESE_CARVE_HISTORY_REF_MARKER   = 'PROCESS_AS_ESE_CARVE_HISTORY_REF_MARKER';
  PROCESS_AS_ESE_CARVE_HISTORY_VISITED      = 'PROCESS_AS_ESE_CARVE_HISTORY_VISITED';
  PROCESS_AS_ESE_TABLES                     = 'PROCESS_AS_ESE_TABLES';
  PROCESS_AS_PLIST          = 'PROCESSASPLIST'; // noslz
  PROCESSALL                = 'PROCESSALL'; // noslz
  RPAD_VALUE                = 55;
  RS_BRAVE                  = 'Brave'; // noslz
  RS_CHROME                 = 'Chrome'; // noslz
  RS_CHROMIUM               = 'Chromium'; // noslz
  RS_EDGE                   = 'Edge'; // noslz
  RS_SIG_JSON               = 'JSON';
  RS_OPERA                  = 'Opera'; // noslz
  RS_FIREFOX                = 'Firefox';
  RS_SAFARI                 = 'Safari'; // noslz
  RS_TOP_SITES              = 'Top Sites'; // noslz
  RUNNING                   = '...';
  SPACE                     = ' ';
  SUM_CHAR                  = '@ ';
  STR_FILES_BY_PATH         = 'Files by path';
  TRIAGE                    = 'TRIAGE'; // noslz
  TSWT                      = 'The script will terminate.';
  VERBOSE                   = False;
  WINDOWS                   = 'Windows'; // noslz

  FORMAT_STR = '%-30s %-30s %-20s %-20s';

var
  gArr:                           TgArr;
  gArr_ValidatedFiles_TList:      array of TList;
  gArtConnect_CatFldr:            TArtifactEntry1;
  gArtConnect_ProgFldr:           array of TArtifactConnectEntry;
  gArtifacts_SQLite:              TSQLite3Database;
  gArtifactsDataStore:            TDataStore;
  gdb_read_bl:                    boolean;
  gFileSystemDataStore:           TDataStore;
  gNumberOfSearchItems:           integer;
  gParameter_Num_StringList:      TStringList;
  gtick_doprocess_i64:            uint64;
  gtick_doprocess_str:            string;
  gtick_foundlist_i64:            uint64;
  gtick_foundlist_str:            string;

  gcol_query:                     TDataStoreField;
  gcol_source_created:            TDataStoreField;
  gcol_source_file:               TDataStoreField;
  gcol_source_modified:           TDataStoreField;
  gcol_source_path:               TDataStoreField;
  gcol_url:                       TDataStoreField;

function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
function FileSubSignatureMatch(Entry: TEntry): boolean;
function GetFullName(Item: PSQL_FileSearch): string;
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
function RPad(const AString: string; AChars: integer): string;
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
function TestForDoProcess(ARefNum: integer): boolean;
function TotalValidatedFileCountInTLists: integer;
procedure ChromePathCheck(aSigShortDisplayName: string; ccEntry: TEntry; indx: integer; aTList: TList; const biTunes_Name_str: string);
procedure DetermineThenSkipOrAdd(Entry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; ref_num: integer; aItems: TSQL_Table_array);

{$IF DEFINED (ISFEXGUI)}
type
  TScriptForm = class(TObject)
  private
    frmMain: TGUIForm;
    FMemo: TGUIMemoBox;
    pnlBottom: TGUIPanel;
    btnOK: TGUIButton;
    btnCancel: TGUIButton;
  public
    ModalResult: boolean;
    constructor Create;
    function ShowModal: boolean;
    procedure OKClick(Sender: TGUIControl);
    procedure CancelClick(Sender: TGUIControl);
    procedure SetText(const Value: string);
    procedure SetCaption(const Value: string);
  end;
{$IFEND}

implementation

{$IF DEFINED (ISFEXGUI)}

constructor TScriptForm.Create;
begin
  inherited Create;
  frmMain := NewForm(nil, SCRIPT_NAME);
  frmMain.Size(500, 480);
  pnlBottom := NewPanel(frmMain, esNone);
  pnlBottom.Size(frmMain.Width, 40);
  pnlBottom.Align(caBottom);
  btnOK := NewButton(pnlBottom, 'OK');
  btnOK.Position(pnlBottom.Width - 170, 4);
  btnOK.Size(75, 25);
  btnOK.Anchor(False, True, True, True);
  btnOK.DefaultBtn := True;
  btnOK.OnClick := OKClick;
  btnCancel := NewButton(pnlBottom, 'Cancel');
  btnCancel.Position(pnlBottom.Width - 88, 4);
  btnCancel.Size(75, 25);
  btnCancel.Anchor(False, True, True, True);
  btnCancel.CancelBtn := True;
  btnCancel.OnClick := CancelClick;
  FMemo := NewMemoBox(frmMain, [eoReadOnly]); // eoNoHScroll, eoNoVScroll
  FMemo.Color := clWhite;
  FMemo.Align(caClient);
  FMemo.FontName('Courier New');
  frmMain.CenterOnParent;
  frmMain.Invalidate;
end;

procedure TScriptForm.OKClick(Sender: TGUIControl);
begin
  // Set variables for use in the main proc. Must do this before closing the main form.
  ModalResult := False;
  ModalResult := True;
  frmMain.Close;
end;

procedure TScriptForm.CancelClick(Sender: TGUIControl);
begin
  ModalResult := False;
  Progress.DisplayMessageNow := CANCELED_BY_USER;
  frmMain.Close;
end;

function TScriptForm.ShowModal: boolean;
begin
  Execute(frmMain);
  Result := ModalResult;
end;

procedure TScriptForm.SetText(const Value: string);
begin
  FMemo.Text := Value;
end;

procedure TScriptForm.SetCaption(const Value: string);
begin
  frmMain.Text := Value;
end;
{$IFEND}

// -----------------------------------------------------------------------------
// Add 40 Character Files
// -----------------------------------------------------------------------------
procedure Add40CharFiles(UniqueList: TUniqueListOfEntries);
var
  Entry: TEntry;
  FileSystemDS: TDataStore;
begin
  FileSystemDS := GetDataStore(DATASTORE_FILESYSTEM);
  if assigned(UniqueList) and assigned(FileSystemDS) then
  begin
    Entry := FileSystemDS.First;
    while assigned(Entry) and Progress.isRunning do
    begin
      begin
        if RegexMatch(Entry.EntryName, '^[0-9a-fA-F]{40}', False) and (Entry.EntryNameExt = '') then // noslz iTunes Backup files
        begin
          UniqueList.Add(Entry);
        end;
      end;
      Entry := FileSystemDS.Next;
    end;
    FreeAndNil(FileSystemDS);
  end;
end;

// -----------------------------------------------------------------------------
// Bookmark Artifact
// -----------------------------------------------------------------------------
procedure Bookmark_Artifact_Source(Entry: TEntry; name_str: string; category_str: string; bm_comment_str: string);
var
  bmDeviceEntry: TEntry;
  bmCreate: boolean;
  bmFolder: TEntry;
  device_name_str: string;
begin
  bmCreate := True;
  if assigned(Entry) then
  begin
    bmDeviceEntry := GetDeviceEntry(Entry);
    device_name_str := '';
    if assigned(bmDeviceEntry) then device_name_str := bmDeviceEntry.SaveName + BS;
    bmFolder := FindBookmarkByName('Artifacts' + BS + 'Source Files' + BS {+ device_name_str + BS} + category_str + BS + name_str, bmCreate);
    if assigned(bmFolder) then
    begin
      if not IsItemInBookmark(bmFolder, Entry) then
      begin
        AddItemsToBookmark(bmFolder, DATASTORE_FILESYSTEM, Entry, bm_comment_str);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Chrome Path Check
// -----------------------------------------------------------------------------
procedure ChromePathCheck(aSigShortDisplayName: string; ccEntry: TEntry; indx: integer; aTList: TList; const biTunes_Name_str: string);
const
  BL_PTH_CHECK_LOGGING = False;
var
  Item: TSQL_FileSearch;
begin
  Item := gArr[indx];

  // Chrome/Chromium
  if (Item.fi_Name_Program = RS_CHROME) or (Item.fi_Name_Program = RS_CHROMIUM) then
  begin
    // Chrome
    if (Item.fi_Name_Program = RS_CHROME) then
    begin
      if (RegexMatch(ccEntry.FullPathName, 'app_chrome\\|\\Chrome\\', False) or RegexMatch(biTunes_Name_str, '\/Chrome\/', False)) and (not RegexMatch(ccEntry.FullPathName, 'Brave|Edge|Opera', False)) then // noslz
      begin
        aTList.Add(ccEntry);
        if BL_PTH_CHECK_LOGGING then
          Progress.Log(RPad(HYPHEN + 'B: Chrome:', RPAD_VALUE) + IntToStr(indx) + ' (' + Item.fi_Name_Program + ')');
      end
    end
    else
    // Chromium
    if (Item.fi_Name_Program = RS_CHROMIUM) then
    begin
      if (not RegexMatch(ccEntry.FullPathName, 'app_chrome\\|\\Chrome\\|Brave|Edge|Opera', False)) and (not RegexMatch(biTunes_Name_str, '\/Chrome|Edge|Opera|android\.chrome', False)) then
      begin
        aTList.Add(ccEntry);
        if BL_PTH_CHECK_LOGGING then
          Progress.Log(RPad('C: Chromium (Other):', RPAD_VALUE) + IntToStr(indx) + ' (' + Item.fi_Name_Program + ')');
      end;
    end;
  end
  else
  begin
    // Brave
    if (Item.fi_Name_Program = RS_BRAVE) and (RegexMatch(ccEntry.FullPathName, 'Brave', False) or RegexMatch(biTunes_Name_str, 'Brave', False)) then // noslz
    begin
      aTList.Add(ccEntry);
    end
    else
    // Edge
    if (Item.fi_Name_Program = RS_EDGE) and (RegexMatch(ccEntry.FullPathName, '\\Edge\\', False) or RegexMatch(biTunes_Name_str, '\/Edge\/', False)) then // noslz
    begin
      aTList.Add(ccEntry);
      if BL_PTH_CHECK_LOGGING then
        Progress.Log(RPad(HYPHEN + 'D: Edge:', RPAD_VALUE) + IntToStr(indx) + ' (' + Item.fi_Name_Program + ')');
      Item.fi_Name_Program := RS_EDGE;
      Item.fi_Icon_Program := ICON_EDGE;
    end
    else
    // Opera
    if (Item.fi_Name_Program = RS_OPERA) and (RegexMatch(ccEntry.FullPathName, 'Opera', False) or RegexMatch(biTunes_Name_str, 'Opera', False)) then // noslz
    begin
      aTList.Add(ccEntry);
      if BL_PTH_CHECK_LOGGING then
        Progress.Log(RPad(HYPHEN + 'E: Opera:', RPAD_VALUE) + IntToStr(indx) + ' (' + Item.fi_Name_Program + ')');
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Clean String
// -----------------------------------------------------------------------------
function CleanString(AString: string): string;
begin
  Result := '';
  AString := StringReplace(AString, '+', ' ', [rfReplaceAll]);
  AString := StringReplace(AString, '%20', ' ', [rfReplaceAll]);
  AString := StringReplace(AString, '%21', '!', [rfReplaceAll]);
  AString := StringReplace(AString, '%22', '"', [rfReplaceAll]);
  AString := StringReplace(AString, '%23', '#', [rfReplaceAll]);
  AString := StringReplace(AString, '%24', '$', [rfReplaceAll]);
  AString := StringReplace(AString, '%25', '%', [rfReplaceAll]);
  AString := StringReplace(AString, '%2520', ' ', [rfReplaceAll]);
  AString := StringReplace(AString, '%26', '&', [rfReplaceAll]);
  AString := StringReplace(AString, '%27', '''', [rfReplaceAll]);
  AString := StringReplace(AString, '%28', '(', [rfReplaceAll]);
  AString := StringReplace(AString, '%29', ')', [rfReplaceAll]);
  AString := StringReplace(AString, '%2A', '*', [rfReplaceAll]);
  AString := StringReplace(AString, '%2B', '+', [rfReplaceAll]);
  AString := StringReplace(AString, '%2C', ',', [rfReplaceAll]);
  AString := StringReplace(AString, '%2D', '-', [rfReplaceAll]);
  AString := StringReplace(AString, '%2E', '.', [rfReplaceAll]);
  AString := StringReplace(AString, '%2F', '/', [rfReplaceAll]);
  AString := StringReplace(AString, '%3A', ':', [rfReplaceAll]);
  AString := StringReplace(AString, '%3B', ';', [rfReplaceAll]);
  AString := StringReplace(AString, '%3C', '<', [rfReplaceAll]);
  AString := StringReplace(AString, '%3D', '=', [rfReplaceAll]);
  AString := StringReplace(AString, '%3E', '>', [rfReplaceAll]);
  AString := StringReplace(AString, '%3F', '?', [rfReplaceAll]);
  AString := StringReplace(AString, '%40', '@', [rfReplaceAll]);
  AString := StringReplace(AString, '%E2%80%93', '-', [rfReplaceAll]);
  AString := StringReplace(AString, '/''/', '', [rfReplaceAll]);
  AString := StringReplace(AString, '''/', '', [rfReplaceAll]);
  Result := AString;
end;

// -----------------------------------------------------------------------------
// Column Value By Name As Date Time
// -----------------------------------------------------------------------------
function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
var
  iCol: integer;
begin
  Result := 0;
  iCol := ColumnByName(Statement, copy(colConf.sql_col, 5, Length(colConf.sql_col)));
  if (iCol > -1) then
  begin
    if colConf.read_as = ftLargeInt then
    try
      Result := Int64ToDateTime_ConvertAs(Statement.Columnint64(iCol), colConf.convert_as);
    except
      on e: exception do
      begin
        Progress.Log(e.message);
      end;
    end
    else if colConf.read_as = ftFloat then
    try
      Result := GHFloatToDateTime(Statement.Columnint64(iCol), colConf.convert_as);
    except
      on e: exception do
      begin
        Progress.Log(e.message);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Count Nodes
// -----------------------------------------------------------------------------
function CountNodes(aNode: TPropertyNode): integer;
var
  aNodeList: TObjectList;
begin
  Result := 0;
  aNodeList := aNode.PropChildList;
  if assigned(aNodeList) then
    Result := aNodeList.Count;
end;

// -----------------------------------------------------------------------------
// Create Artifact Description Text
// -----------------------------------------------------------------------------
function Create_Artifact_Description_Text(Item: TSQL_FileSearch): string;
var
  aStringList: TStringList;

  procedure AddIntToSL(const name_str: string; aint: integer);
  begin
    if aint > 0 then
      aStringList.Add(RPad(name_str + ':', 25) + IntToStr(aint));
  end;

  procedure AddStrToSL(const name_str: string;const astr: string);
  begin
    if Trim(astr) <> '' then
      aStringList.Add(RPad(name_str + ':', 25) + astr);
  end;

begin
  Result := '';
  aStringList := TStringList.Create;
  try
    AddStrToSL('Process_ID', Item.fi_Process_ID);
    AddStrToSL('Name_Program', Item.fi_Name_Program);
    AddStrToSL('Name_Program_Type', Item.fi_Name_Program_Type);
    AddStrToSL('Name_OS', Item.fi_Name_OS);
    AddStrToSL('Process_As', Item.fi_Process_As);
    AddIntToSL('Carve_Adjustment', Item.fi_Carve_Adjustment);
    AddStrToSL('Carve_Footer', Item.fi_Carve_Footer);
    AddStrToSL('Carve_Header', Item.fi_Carve_Header);
    AddIntToSL('Icon_Category', Item.fi_Icon_Category);
    AddIntToSL('Icon_OS', Item.fi_Icon_OS);
    AddIntToSL('Icon_Program', Item.fi_Icon_Program);
    AddStrToSL('NodeByName', Item.fi_NodeByName);
    AddStrToSL('Reference_Info', Item.fi_Reference_Info);
    AddStrToSL('Rgx_Itun_Bkup_Dmn', Item.fi_Rgx_Itun_Bkup_Dmn);
    AddStrToSL('Rgx_Itun_Bkup_Nme', Item.fi_Rgx_Itun_Bkup_Nme);
    AddStrToSL('Glob1_Search', Item.fi_Glob1_Search);
    AddStrToSL('Glob2_Search', Item.fi_Glob2_Search);
    AddStrToSL('Glob3_Search', Item.fi_Glob3_Search);
    AddStrToSL('Regex_Search', Item.fi_Regex_Search);
    AddStrToSL('RootNodeName', Item.fi_RootNodeName);
    AddStrToSL('Signature_Parent', Item.fi_Signature_Parent);
    AddStrToSL('Signature_Sub', Item.fi_Signature_Sub);
    AddStrToSL('SQLStatement', Item.fi_SQLStatement);
    AddStrToSL('fi_SQLTables_Required', Item.fi_SQLTables_Required);
    AddStrToSL('SQLPrimary_Tablestr', Item.fi_SQLPrimary_Tablestr);
    Result := aStringList.Text;
  finally
    aStringList.free;
  end;
end;

// -----------------------------------------------------------------------------
// Create Global Search
// -----------------------------------------------------------------------------
procedure Create_Global_Search(anint: integer; Item: TSQL_FileSearch; aStringList: TStringList);
var
  Glob1_Search: string;
  Glob2_Search: string;
  Glob3_Search: string;
begin
  Glob1_Search := Item.fi_Glob1_Search;
  Glob2_Search := Item.fi_Glob2_Search;
  Glob3_Search := Item.fi_Glob3_Search;
  if Trim(Glob1_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob1_Search + ''');'); // noslz
  if Trim(Glob2_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob2_Search + ''');'); // noslz
  if Trim(Glob3_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob3_Search + ''');'); // noslz
  if Trim(Glob1_Search) =  '' then Progress.Log(IntTostr(anint) + ': No Glob Search'); // noslz
end;

// -----------------------------------------------------------------------------
// Determine Then Skip Or Add
// -----------------------------------------------------------------------------
procedure DetermineThenSkipOrAdd(Entry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
var
  bm_comment_str: string;
  bmwal_Entry: TEntry;
  DeterminedFileDriverInfo: TFileTypeInformation;
  File_Added_bl: boolean;
  first_Entry: TEntry;
  i: integer;
  Item: TSQL_FileSearch;
  MatchSignature_str: string;
  NowProceed_bl: boolean;
  reason_str: string;
  Reason_StringList: TStringList;
  trunc_EntryName_str: string;

begin
  File_Added_bl := False;
  if Entry.isSystem and ((POS('$I30', Entry.EntryName) > 0) or (POS('$90', Entry.EntryName) > 0)) then
  begin
    NowProceed_bl := False;
    Exit;
  end;

  DeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
  reason_str := '';
  trunc_EntryName_str := copy(Entry.EntryName, 1, 25);
  Reason_StringList := TStringList.Create;
  try
    for i := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      NowProceed_bl := False;
      reason_str := '';
      Item := gArr[i];

      // -------------------------------------------------------------------------------
      // Special validation (these files are still sent here via the Regex match)
      // -------------------------------------------------------------------------------
      if (not NowProceed_bl) then
      begin
        if (Item.fi_Process_As = PROCESS_AS_ESE_CARVE_CONTENT) and RegexMatch(Entry.EntryName, '^V01.*\.log', False) then
          NowProceed_bl := True;

        if (Item.fi_Process_As = PROCESS_AS_ESE_CARVE_HISTORY_VISITED) and RegexMatch(Entry.EntryName, '^V01.*\.log', False) then
          NowProceed_bl := True;
      end;

      // Proceed if SubDriver has been identified
      if (not NowProceed_bl) then
      begin
        if Item.fi_Signature_Sub <> '' then
        begin
          if RegexMatch(RemoveSpecialChars(DeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then
          begin
            NowProceed_bl := True;
            reason_str := 'ShortDisplayName = Required SubSig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + Item.fi_Signature_Sub + ')';
            Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(A)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
          end;
        end;
      end;

      // Set the MatchSignature to the parent
      if (not NowProceed_bl) and not(UpperCase(Entry.Extension) = '.JSON') then
      begin
        MatchSignature_str := '';
        if Item.fi_Signature_Sub = '' then
          MatchSignature_str := UpperCase(Item.fi_Signature_Parent);
        // Proceed if SubDriver is blank, but File/Path Name and Parent Signature match
        if ((RegexMatch(Entry.EntryName, Item.fi_Regex_Search, False)) or (RegexMatch(Entry.FullPathName, Item.fi_Regex_Search, False))) and
          ((UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) or (RegexMatch(DeterminedFileDriverInfo.ShortDisplayName, MatchSignature_str, False))) then
        begin
          NowProceed_bl := True;
          reason_str := 'ShortDisplay matches Parent sig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
          Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(B)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
        end;
      end;

      // Proceed if EntryName is unknown, but iTunes Domain, Name and Sig match
      if (not NowProceed_bl) then
      begin
        if RegexMatch(biTunes_Domain_str, Item.fi_Rgx_Itun_Bkup_Dmn, False) and RegexMatch(biTunes_Name_str, Item.fi_Rgx_Itun_Bkup_Nme, False) and (UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) then
        begin
          NowProceed_bl := True;
          reason_str := 'Proceed on Sig and iTunes Domain\Name:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
          Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(C)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
        end;
      end;

      if NowProceed_bl then
      begin
        // Chrome path check ---------------------------------------------------

        // Chrome Path Check - Autofill
        if (DeterminedFileDriverInfo.ShortDisplayName = RS_SIG_CHROME_AUTOFILL) then
          ChromePathCheck(RS_SIG_CHROME_AUTOFILL, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
        else

          // Chrome Path Check - Bookmarks
          if (DeterminedFileDriverInfo.ShortDisplayName = RS_SIG_JSON) then
            ChromePathCheck(RS_SIG_JSON, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
          else

            // Chrome Path Check - Cache
            if (DeterminedFileDriverInfo.ShortDisplayName = RS_SIG_CHROME_CACHE_DATA) then
              ChromePathCheck(RS_SIG_CHROME_CACHE_DATA, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
            else

              // Chrome Path Check - Chrome Cookies
              if (DeterminedFileDriverInfo.ShortDisplayName = RS_SIG_CHROME_COOKIES) then
                ChromePathCheck(RS_SIG_CHROME_COOKIES, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
              else

                // Chrome Path Check - Fav Icons
                if (DeterminedFileDriverInfo.ShortDisplayName = RS_SIG_CHROME_FAVICONS) then
                  ChromePathCheck(RS_SIG_CHROME_FAVICONS, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
                else

                  // Chrome Path Check - History
                  if (DeterminedFileDriverInfo.ShortDisplayName = RS_SIG_CHROME_HISTORY) then
                    ChromePathCheck(RS_SIG_CHROME_HISTORY, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
                  else

                    // Chrome Path Check - Logins
                    if (DeterminedFileDriverInfo.ShortDisplayName = RS_SIG_CHROME_LOGINS) then
                      ChromePathCheck(RS_SIG_CHROME_LOGINS, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
                    else

                      // Chrome Path Check - Shortcuts
                      if (DeterminedFileDriverInfo.ShortDisplayName = RS_SIG_CHROME_SHORTCUTS) then
                        ChromePathCheck(RS_SIG_CHROME_SHORTCUTS, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
                      else

                        // Chrome Path Check - Top Sites
                        if (Item.fi_Name_Program_Type = RS_TOP_SITES) and (DeterminedFileDriverInfo.ShortDisplayName = RS_SIG_SQLITE) then
                          ChromePathCheck(RS_SIG_SQLITE, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
                        else

                          // Chrome Path Check - Top Sites (Thmbs)
                          if (Item.fi_Name_Program_Type = RS_TOP_SITES + SPACE + '(Thmbs)') and (DeterminedFileDriverInfo.ShortDisplayName = 'Chrome Topsites') then // noslz
                            ChromePathCheck(RS_SIG_SQLITE, Entry, i, gArr_ValidatedFiles_TList[i], biTunes_Name_str)
                          else

                            // Firefox Path Check -----------------------------------------------------
                            if Item.fi_Process_ID = 'BRW_FIREFOX_CACHE' then
                            begin
                              if RegexMatch(Entry.FullPathName, 'firefox|mozilla', False) and (DeterminedFileDriverInfo.ShortDisplayName = 'URL Cache') then // noslz
                              begin
                                gArr_ValidatedFiles_TList[i].Add(Entry);
                              end
                              else
                                File_Added_bl := False;
                            end
                            else

                              // Safari Path Check -----------------------------------------------------
                              if Item.fi_Process_ID = 'BRW_SAFARI_CACHE' then
                              begin
                                if RegexMatch(Entry.FullPathName, 'Safari', False) and (DeterminedFileDriverInfo.ShortDisplayName = 'URL Cache') then // noslz
                                begin
                                  gArr_ValidatedFiles_TList[i].Add(Entry);
                                end
                                else
                                  File_Added_bl := False;
                              end

                              // Everything Else Gets Added Here ---------------------------------------
                              else
                              begin
                                gArr_ValidatedFiles_TList[i].Add(Entry);
                              end;

        File_Added_bl := True;

        // Bookmark Source Files
        if BL_BOOKMARK_SOURCE then
        begin
          bm_comment_str := Create_Artifact_Description_Text(gArr[i]);
          Bookmark_Artifact_Source(Entry, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, CATEGORY_NAME, bm_comment_str);
          if assigned(gFileSystemDataStore) then
          begin
            first_Entry := gFileSystemDataStore.First;
            if assigned(first_Entry) then
            begin
              bmwal_Entry := gFileSystemDataStore.FindByPath(first_Entry, Entry.FullPathName + '-wal');
              if bmwal_Entry <> nil then
              begin
                Bookmark_Artifact_Source(bmwal_Entry, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, CATEGORY_NAME, bm_comment_str);
              end;
            end;
          end;
        end;

        // Add Flags
        if BL_USE_FLAGS then Entry.Flags := Entry.Flags + [Flag5]; // Green Flag
      end;
    end;

    if NOT(File_Added_bl) then
      Reason_StringList.Add(format(GFORMAT_STR, ['', 'Ignored', '', IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]));

    for i := 0 to Reason_StringList.Count - 1 do
      Progress.Log(Reason_StringList[i]);

  finally
    Reason_StringList.free;
  end;
end;

// -----------------------------------------------------------------------------
// File Sub Signature Match
// -----------------------------------------------------------------------------
function FileSubSignatureMatch(Entry: TEntry): boolean;
var
  i: integer;
  param_num_int: integer;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  Item: TSQL_FileSearch;
begin
  Result := False;
  if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
  begin
    for i := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      Item := gArr[i];
      if Item.fi_Signature_Sub <> '' then
      begin
        aDeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
        if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
        begin
          if BL_PROCEED_LOGGING then
            Progress.Log(RPad('Proceed' + HYPHEN + 'Identified by SubSig:', RPAD_VALUE) + Entry.EntryName + SPACE + 'Bates:' + IntToStr(Entry.ID));
          Result := True;
          break;
        end;
      end;
    end;
  end
  else
  begin
    if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
    begin
      for i := 0 to gParameter_Num_StringList.Count - 1 do
      begin
        if not Progress.isRunning then
          break;
        param_num_int := StrToInt(gParameter_Num_StringList[i]);
        Item := gArr[param_num_int];
        if Item.fi_Signature_Sub <> '' then
        begin
          aDeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
          if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
          begin
            if BL_PROCEED_LOGGING then
              Progress.Log(RPad('Proceed' + HYPHEN + 'File Sub-Signature Match:', RPAD_VALUE) + Entry.EntryName + SPACE + 'Bates:' + IntToStr(Entry.ID));
            Result := True;
            break;
          end;
        end;
      end
    end
  end;
end;

// -----------------------------------------------------------------------------
// Get Full Name
// -----------------------------------------------------------------------------
function GetFullName(Item: PSQL_FileSearch): string;
var
  ApplicationName: string;
  TypeName: string;
  OSName: string;
begin
  Result := '';
  ApplicationName := Item.fi_Name_Program;
  TypeName := Item.fi_Name_Program_Type;
  OSName := Item.fi_Name_OS;
  if (ApplicationName <> '') then
  begin
    if (TypeName <> '') then
      Result := format('%0:s %1:s', [ApplicationName, TypeName])
    else
      Result := ApplicationName;
  end
  else
    Result := TypeName;
  if OSName <> '' then
    Result := Result + ' ' + OSName;
end;

// -----------------------------------------------------------------------------
// Length of Array Table
// -----------------------------------------------------------------------------
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
var
  i: integer;
begin
  Result := 0;
  for i := 1 to 100 do
  begin
    if anArray[i].sql_col = '' then
      break;
    Result := i;
  end;
end;

// -----------------------------------------------------------------------------
// RPad
// -----------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString);
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

// -----------------------------------------------------------------------------
// Setup Column For Folder
// -----------------------------------------------------------------------------
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
var
  col_label: string;
  Field: TDataStoreField;
  i: integer;
  Item: TSQL_FileSearch;
  NumberOfColumns: integer;

begin
  Result := True;
  Item := gArr[aReferenceNumber];
  NumberOfColumns := ColCount;
  SetLength(col_DF, ColCount + 1);

  if assigned(anArtifactFolder) then
  begin
    for i := 1 to NumberOfColumns do
    begin
      try
        if not Progress.isRunning then
          Exit;
        Field := gArtifactsDataStore.DataFields.FieldByName(aItems[i].fex_col);
        if assigned(Field) and (Field.FieldType <> aItems[i].col_type) then
        begin
          MessageUser(SCRIPT_NAME + DCR + 'WARNING: New column: ' + DCR + aItems[i].fex_col + DCR + 'already exists as a different type. Creation skipped.');
          Result := False;
        end
        else
        begin
          col_label := '';
          col_DF[i] := gArtifactsDataStore.DataFields.Add(aItems[i].fex_col + col_label, aItems[i].col_type);
          if col_DF[i] = nil then
          begin
            MessageUser(SCRIPT_NAME + DCR + 'Cannot use a fixed field. Please contact support@getdata.com quoting the following error: ' + DCR + SCRIPT_NAME + SPACE + IntToStr(aReferenceNumber) + SPACE + aItems[i].fex_col);
            Result := False;
          end;
        end;
      except
        MessageUser(ATRY_EXCEPT_STR + 'Failed to create column');
      end;
    end;

    // Set the Source Columns --------------------------------------------------
    gcol_source_file := gArtifactsDataStore.DataFields.GetFieldByName('Source_Name');
    gcol_source_path := gArtifactsDataStore.DataFields.GetFieldByName('Source_Path');
    gcol_source_created := gArtifactsDataStore.DataFields.GetFieldByName('Source_Created');
    gcol_source_modified := gArtifactsDataStore.DataFields.GetFieldByName('Source_Modified');
    gcol_url := gArtifactsDataStore.DataFields.GetFieldByName('URL');
    gcol_query := gArtifactsDataStore.DataFields.GetFieldByName('Query');

    // Columns -----------------------------------------------------------------
    if Result then
    begin
      // Enables the change of column headers when switching folders - This is the order of displayed columns
      for i := 1 to NumberOfColumns do
      begin
        if not Progress.isRunning then
          break;
        if aItems[i].Show then
        begin
          // Progress.Log('Add Field Name: ' + col_DF[i].FieldName);
          anArtifactFolder.AddField(col_DF[i]);
        end;
      end;

      if (Item.fi_Process_As = 'POSTPROCESS') then
        anArtifactFolder.AddField(gcol_source_path)
      else
      begin
        anArtifactFolder.AddField(gcol_source_file);
        anArtifactFolder.AddField(gcol_source_path);
        anArtifactFolder.AddField(gcol_source_created);
        anArtifactFolder.AddField(gcol_source_modified);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Name
// -----------------------------------------------------------------------------
function SQLColumnByName(Statement: TSQLite3Statement; const Name: string): integer;
var
  i: integer;
begin
  Result := -1;
  for i := 0 to Statement.ColumnCount do
  begin
    if not Progress.isRunning then
      break;
    if SameText(Statement.ColumnName(i), Name) then
    begin
      Result := i;
      break;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Integer
// -----------------------------------------------------------------------------
function SQLColumnValueByNameAsInt(Statement: TSQLite3Statement; const Name: string): integer;
var
  iCol: integer;
begin
  Result := -1;
  iCol := SQLColumnByName(Statement, Name);
  if iCol > -1 then
    Result := Statement.ColumnInt(iCol);
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Text
// -----------------------------------------------------------------------------
function SQLColumnValueByNameAsText(Statement: TSQLite3Statement; const Name: string): string;
var
  iCol: integer;
begin
  Result := '';
  iCol := SQLColumnByName(Statement, Name);
  if iCol > -1 then
    Result := Statement.ColumnText(iCol);
end;

// -----------------------------------------------------------------------------
// SQL - Read Artifacts DB
// -----------------------------------------------------------------------------
procedure Read_SQLite_DB();
var
  i: integer;
  sql_db_path_str: string;
  sqlselect: TSQLite3Statement;

begin
  // Locate the SQLite DB
  sql_db_path_str := GetDatabasesDir + 'Artifacts' + BS + ARTIFACTS_DB; // noslz
  if not FileExists(sql_db_path_str) then
  begin
    MessageUser('Browser' + ':' + SPACE + 'Did not locate Artifacts SQLite Database:' + SPACE + sql_db_path_str + '.' + DCR + TSWT);
    Exit;
  end;
  Progress.Log(RPad('Found Database:', RPAD_VALUE) + sql_db_path_str);

  // Open the database
  gArtifacts_SQLite := TSQLite3Database.Create;
  try
    gdb_read_bl := False;
    if FileExists(sql_db_path_str) then
    try
      gArtifacts_SQLite.Open(sql_db_path_str);
      gdb_read_bl := True;
      Progress.Log(RPad('Database Read:', RPAD_VALUE) + BoolToStr(gdb_read_bl, True));
    except
      on e: exception do
      begin
        Progress.Log(e.message);
        Exit;
      end;
    end;

    // Get the number of rows in the database as gNumberOfSearchItems
    if gdb_read_bl then
    begin
      sqlselect := TSQLite3Statement.Create(gArtifacts_SQLite, 'SELECT COUNT(*) FROM Artifact_Values');
      try
        gNumberOfSearchItems := 0;
        if sqlselect.Step = SQLITE_ROW then
          gNumberOfSearchItems := sqlselect.ColumnInt(0);
      finally
        sqlselect.free;
      end;
      Progress.Log(RPad('Database Rows:', RPAD_VALUE) + IntToStr(gNumberOfSearchItems));

      if gNumberOfSearchItems = 0 then
      begin
        MessageUser('No records were read from:' + SPACE + sql_db_path_str + '.' + DCR + TSWT);
        Exit;
      end;
      Progress.Log(StringOfChar('-', CHAR_LENGTH));

      SetLength(gArr, gNumberOfSearchItems);

      // Populate the Array with values from the database
        sqlselect := TSQLite3Statement.Create(gArtifacts_SQLite, 'SELECT * FROM Artifact_Values ORDER BY CASE WHEN fi_Name_Program LIKE ''@%'' THEN 1 ELSE 0 END, fi_Name_Program;');
        try
        i := 0;
        while (sqlselect.Step = SQLITE_ROW) and (Progress.isRunning) do
        begin
          gArr[i].fi_Carve_Adjustment    := SQLColumnValueByNameAsInt(sqlselect,  'fi_Carve_Adjustment');
          gArr[i].fi_Carve_Footer        := SQLColumnValueByNameAsText(sqlselect, 'fi_Carve_Footer');
          gArr[i].fi_Carve_Header        := SQLColumnValueByNameAsText(sqlselect, 'fi_Carve_Header');
          gArr[i].fi_ESE_ContainerName   := SQLColumnValueByNameAsText(sqlselect, 'fi_ESE_ContainerName');
          gArr[i].fi_Glob1_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob1_Search');
          gArr[i].fi_Glob2_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob2_Search');
          gArr[i].fi_Glob3_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob3_Search');
          gArr[i].fi_Icon_Category       := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_Category');
          gArr[i].fi_Icon_OS             := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_OS');
          gArr[i].fi_Icon_Program        := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_Program');
          gArr[i].fi_Name_Program        := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_Program');
          gArr[i].fi_Name_OS             := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_OS');
          gArr[i].fi_Name_Program_Type   := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_Program_Type');
          gArr[i].fi_NodeByName          := SQLColumnValueByNameAsText(sqlselect, 'fi_NodeByName');
          gArr[i].fi_Process_As          := SQLColumnValueByNameAsText(sqlselect, 'fi_Process_As');
          gArr[i].fi_Process_ID          := SQLColumnValueByNameAsText(sqlselect, 'fi_Process_ID');
          gArr[i].fi_Reference_Info      := SQLColumnValueByNameAsText(sqlselect, 'fi_Reference_Info');
          gArr[i].fi_Regex_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Regex_Search');
          gArr[i].fi_Rgx_Itun_Bkup_Dmn   := SQLColumnValueByNameAsText(sqlselect, 'fi_Rgx_Itun_Bkup_Dmn');
          gArr[i].fi_Rgx_Itun_Bkup_Nme   := SQLColumnValueByNameAsText(sqlselect, 'fi_Rgx_Itun_Bkup_Nme');
          gArr[i].fi_RootNodeName        := SQLColumnValueByNameAsText(sqlselect, 'fi_RootNodeName');
          gArr[i].fi_Signature_Parent    := SQLColumnValueByNameAsText(sqlselect, 'fi_Signature_Parent');
          gArr[i].fi_Signature_Sub       := SQLColumnValueByNameAsText(sqlselect, 'fi_Signature_Sub');
          gArr[i].fi_SQLPrimary_Tablestr := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLPrimary_Tablestr');
          gArr[i].fi_SQLStatement        := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLStatement');
          gArr[i].fi_SQLTables_Required  := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLTables_Required');
          gArr[i].fi_Test_Data           := SQLColumnValueByNameAsText(sqlselect, 'fi_Test_Data');
          inc(i);
        end;

      finally
        sqlselect.free;
      end;

    end;

  finally
    gArtifacts_SQLite.Close;
    gArtifacts_SQLite.free;
  end;
end;

// -----------------------------------------------------------------------------
// StrippedOfNonAscii
// -----------------------------------------------------------------------------
function StrippedOfNonAscii(const str: string): string;
var
  idx, Count: integer;
begin
  SetLength(Result, Length(str));
  Count := 0;
  for idx := 1 to Length(str) do
  begin
    if ((str[idx] >= #32) and (str[idx] <= #127)) or (str[idx] in [#10, #13]) then
    begin
      Inc(Count);
      Result[Count] := str[idx];
    end;
  end;
  SetLength(Result, Count);
end;

// -----------------------------------------------------------------------------
// Test for Do Process
// -----------------------------------------------------------------------------
function TestForDoProcess(ARefNum: integer): boolean;
begin
  Result := False;

  if (ARefNum <= Length(gArr) - 1) and Progress.isRunning then
  begin
    if (CmdLine.Params.Indexof(IntToStr(ARefNum)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      if gArr_ValidatedFiles_TList[ARefNum].Count > 0 then
      begin
        Progress.Log(RPad('Process List #' + IntToStr(ARefNum) + SPACE + '(' + IntToStr(gArr_ValidatedFiles_TList[ARefNum].Count) + '):', RPAD_VALUE) + gArr[ARefNum].fi_Name_Program + SPACE + gArr[ARefNum].fi_Name_Program_Type + RUNNING);
        if gArr[ArefNum].fi_Process_ID <> '' then  Progress.Log(RPad('Process ID:', RPAD_VALUE) + gArr[ARefNum].fi_Process_ID);
        if gArr[ARefNum].fi_Process_As <> '' then Progress.Log(RPad('fi_Process_As:', RPAD_VALUE) + gArr[ARefNum].fi_Process_As);
        if gArr[ARefNum].fi_ESE_ContainerName <> '' then Progress.Log(RPad('fi_ESE_ContainerName:', RPAD_VALUE)+ gArr[ARefNum].fi_ESE_ContainerName);
        Result := True;
      end;
    end;
  {$IF DEFINED (ISFEXGUI)}
  end
  else
  begin
    if not Progress.isRunning then
      Exit;
    Progress.Log('Error: RefNum > ' + IntToStr(Length(gArr) - 1)); // noslz
  {$IFEND}
  end;
end;

// -----------------------------------------------------------------------------
// Total Validated File Count
// -----------------------------------------------------------------------------
function TotalValidatedFileCountInTLists: integer;
var
  i: integer;
begin
  Result := 0;
  for i := 0 to Length(gArr) - 1 do
  begin
    if not Progress.isRunning then
      break;
    Result := Result + gArr_ValidatedFiles_TList[i].Count;
  end;
end;


// Function: Carve Date ------------------------------------------------------
// From \xFE.. backwards: (Regex match starts here)
// - 18 back to end of first date  = 18
// - 08 Access Time                = 26
// - 08 Modified Time              = 34
// - 08 Expiry Time                = 42
// - 08 Creation Time              = 50
// - 08 Sync Time                  = 58
// - 04 start of access count      = 62
// - 04 flag                       = 66
// - 24 URLHash                    = 90
// - total

  procedure PrintCarveHeader(hdr_start: int64; hdr_end: int64; hdr_count: int64);
  begin
    if VERBOSE then
      Progress.Log((format('%-29s %-20s %-20s %-20s', ['Regex Match: Visited', 'Start', 'End', 'Total bytes'])));
    if VERBOSE then
      Progress.Log((format('%-29s %-20s %-20s %-20s', ['', IntToStr(hdr_start), IntToStr(hdr_end), IntToStr(hdr_count)])));
    if VERBOSE then
      Progress.Log(' ');
  end;

// =============================================================================
// Do Process
// =============================================================================
procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; ref_num: integer; aItems: TSQL_Table_array);
var
  aArtifactEntry: TEntry;
  ADDList: TList;
  aPropertyTree: TPropertyParent;
  aRootProperty: TPropertyNode;
  b, i, g, x, y, z: integer;
  ByteArray: array of byte;
  CarvedData: TByteInfo;
  CarvedEntry: TEntry;
  CarvedEntryReader: TEntryReader;
  carved_str: string;
  ColCount: integer;
  col_DF: TDataStoreFieldArray;
  curpos: integer;
  data_size: int64;
  Display_Name_str: string;
  DNT_sql_col: string;
  dt_str: string;
  end_pos: int64;
  FooterProgress: TPAC;
  FooterReader: TEntryReader;
  gh_int: integer;
  HeaderReader: TEntryReader;
  HeaderRegex, FooterRegEx: TRegEx;
  h_startpos, h_offset, h_count, f_offset, f_count: int64;
  Item: PSQL_FileSearch;
  Log_ChromeCache_bl: boolean;
  mydb: TSQLite3Database;
  newEntryReader: TEntryReader;
  newJOURNALReader: TEntryReader;
  newWALReader: TEntryReader;
  Node1, Node2, Node3: TPropertyNode;
  NodeList1, NodeList2: TObjectList;
  NumberOfNodes: integer;
  records_read_int: integer;
  refetch_count: int64;
  reuse_count: int64;
  Size: integer;
  sqlselect: TSQLite3Statement;
  sql_row_count: integer;
  sSize: integer;
  startpos: integer;
  Temp_NodeList: TList;
  TestNode1: TPropertyNode;
  TestNode2: TPropertyNode;
  test_bytes: Tbytes;
  TotalFiles: int64;
  url_dt: TDateTime;
  url_len: dword;
  url_str, title_str, fav_str: string;
  url_unixdt: int64;
  variant_Array: array of variant;

  procedure NullTheArray;
  var
    idx: integer;
  begin
    for idx := 1 to ColCount do
      variant_Array[idx] := null;
  end;

// ▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼
  procedure AddToModule;
  var
    NEntry: TArtifactItem;
    populate_bl, IsAllEmpty: boolean;
  begin
    populate_bl := False;
    // Do not add when it is a Triage
    if CmdLine.Params.Indexof(TRIAGE) > -1 then
      Exit;

    // Check if all columns are empty
    IsAllEmpty := True;
    for g := 1 to ColCount do
    begin
      if (not VarIsEmpty(variant_Array[g])) and (not VarIsNull(variant_Array[g])) then
      begin
        IsAllEmpty := False;
        break;
      end;
    end;

    // If artifacts are found
    if not IsAllEmpty then
    begin

      // Create the Category Folder to the tree --------------------------------
      if not assigned(gArtConnect_CatFldr) then
      begin
        gArtConnect_CatFldr := AddArtifactCategory(nil, CATEGORY_NAME, -1, gArr[1].fi_Icon_Category); { Sort index, icon }
        gArtConnect_CatFldr.Status := gArtConnect_CatFldr.Status + [dstUserCreated];
      end;

      // Create artifact sub-folder
      gArtConnect_ProgFldr[ref_num] := AddArtifactConnect(TEntry(gArtConnect_CatFldr),
      Item.fi_Name_Program,
      Item.fi_Name_Program_Type,
      Item.fi_Name_OS,
      Item.fi_Icon_Program,
      Item.fi_Icon_OS);

      // If the artifact sub-folder has been created
      if assigned(gArtConnect_ProgFldr[ref_num]) then
      begin
        // Set the status of the artifacts sub-folder
        gArtConnect_ProgFldr[ref_num].Status := gArtConnect_ProgFldr[ref_num].Status + [dstUserCreated];

        // Setup the columns of the artifact sub-folder
        SetUpColumnforFolder(ref_num, gArtConnect_ProgFldr[ref_num], col_DF, ColCount, aItems);

        // Create the new entry
        NEntry := TArtifactItem.Create;
        NEntry.SourceEntry := aArtifactEntry;
        NEntry.Parent := gArtConnect_ProgFldr[ref_num];
        NEntry.PhysicalSize := 0;
        NEntry.LogicalSize := 0;

        // Populate the columns
        try
          for g := 1 to ColCount do
          begin
            if not Progress.isRunning then
              break;

            if (VarIsNull(variant_Array[g])) or (VarIsEmpty(variant_Array[g])) then
              Continue;
			  
            if RegexMatch(Item^.fi_Process_ID, 'BRW_IE_49_', True) then
              if UpperCase(Item^.fi_Name_Program_Type) = UpperCase(variant_Array[g]) then
                populate_bl := True;

            case col_DF[g].FieldType of

              ftDateTime:
                if not VarIsStr(variant_Array[g]) then
                  try
                    col_DF[g].AsDateTime[NEntry] := variant_Array[g];
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                      Progress.Log(HYPHEN + 'ftDateTime conversion');
                    end;
                  end;

              ftFloat:
                if VarIsStr(variant_Array[g]) and (variant_Array[g] <> '') then
                  try
                    col_DF[g].AsFloat[NEntry] := StrToFloat(variant_Array[g]);
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                      Progress.Log(HYPHEN + 'ftFloat conversion');
                    end;
                  end;

              ftInteger:
                try
                  if Trim(variant_Array[g]) = '' then
                    variant_Array[g] := null
                  else
                    col_DF[g].AsInteger[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftInteger conversion');
                  end;
                end;

              ftLargeInt:
                try
                  col_DF[g].AsInt64[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftLargeInt conversion');
                  end;
                end;

              ftString:
                if VarIsStr(variant_Array[g]) and (variant_Array[g] <> '') then
                  try
                    col_DF[g].AsString[NEntry] := variant_Array[g];
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                      Progress.Log(HYPHEN + 'ftString conversion');
                    end;
                  end;

              ftBytes:
                try
                  col_DF[g].AsBytes[NEntry] := variantToArrayBytes(variant_Array[g]);
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftBytes conversion');
                  end;
                end;

            end;
          end;
        except
          on e: exception do
          begin
            Progress.Log(e.message);
          end;
        end;

        if (RegexMatch(Item.fi_Process_ID, 'BRW_IE_49_', True)) and populate_bl then
          ADDList.Add(NEntry);
        if not(RegexMatch(Item.fi_Process_ID, 'BRW_IE_49_', True)) then
          ADDList.Add(NEntry);

      end;
    end;
    NullTheArray;
  end;
// ▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲

  function Test_SQL_Tables(refnum: integer): boolean;
  var
    LineList: TStringList;
    idx: integer;
    sql_tbl_count: integer;
    sqlselect_row: TSQLite3Statement;
    s: integer;
    temp_sql_str: string;

  begin
    Result := True;

    // Table count check to eliminate corrupt SQL files with no tables
    sql_tbl_count := 0;
    try
      sqlselect := TSQLite3Statement.Create(mydb, 'SELECT name FROM sqlite_master WHERE type=''table''');
      try
        while sqlselect.Step = SQLITE_ROW do
        begin
          if not Progress.isRunning then
            break;
          sql_tbl_count := sql_tbl_count + 1;
        end;
      finally
        FreeAndNil(sqlselect);
      end;
    except
      Progress.Log(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
      Result := False;
      Exit;
    end;

    if sql_tbl_count = 0 then
    begin
      Result := False;
      Exit;
    end;

    // Special check for Safari Bookmarks to exclude Safari Tabs
    if (Item.fi_Name_Program = 'Safari') and (Item.fi_Name_Program_Type = 'Bookmarks') then // noslz
    begin
      if sql_tbl_count > 7 then {Boomkarks has 7, Safari Tabs has 9}
      begin
        Result := False;
        Exit;
      end;
    end;

    // Table count check to eliminate SQL files without required tables
    if (sql_tbl_count > 0) and (gArr[refnum].fi_SQLTables_Required <> '') then
    begin
      LineList := TStringList.Create;
      LineList.Delimiter := ','; // Default, but ";" is used with some locales
      LineList.StrictDelimiter := True; // Required: strings are separated *only* by Delimiter
      LineList.Sorted := True;
      LineList.Duplicates := dupIgnore;
      try
        LineList.DelimitedText := gArr[refnum].fi_SQLTables_Required;
        if LineList.Count > 0 then
        begin
          temp_sql_str := 'select count(*) from sqlite_master where type = ''table'' and ((upper(name) = upper(''' + Trim(LineList[0]) + '''))';
          for idx := 0 to LineList.Count - 1 do
            temp_sql_str := temp_sql_str + ' or (upper(name) = upper(''' + Trim(LineList[idx]) + '''))';

          temp_sql_str := temp_sql_str + ')';
          try
            sqlselect_row := TSQLite3Statement.Create(mydb, temp_sql_str);
            try
              if (sqlselect_row.Step = SQLITE_ROW) and (sqlselect_row.ColumnInt(0) = LineList.Count) then
                sql_tbl_count := 1
              else
              begin
                Progress.Log(RPad(HYPHEN + 'Bates:' + SPACE + IntToStr(aArtifactEntry.ID) + HYPHEN + aArtifactEntry.EntryName, RPAD_VALUE) + 'Ignored (Found ' + IntToStr(sqlselect_row.ColumnInt(0)) + ' of ' + IntToStr(LineList.Count) + ' required tables).');
                Result := False;
              end;
            finally
              sqlselect_row.free;
            end;
          except
            MessageUser(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
            Result := False;
          end;
        end;

        // Log the missing required tables
        for s := 0 to LineList.Count - 1 do
        begin
          temp_sql_str := 'select count(*) from sqlite_master where type = ''table'' and ((upper(name) = upper(''' + Trim(LineList[s]) + ''')))';
          try
            sqlselect_row := TSQLite3Statement.Create(mydb, temp_sql_str);
            try
              if (sqlselect_row.Step = SQLITE_ROW) then
              begin
                if sqlselect_row.ColumnInt(0) = 0 then
                begin
                  Progress.Log(RPad('', RPAD_VALUE) + HYPHEN + 'Missing table:' + SPACE + Trim(LineList[s]));
                end;
              end;
            finally
              sqlselect_row.free;
            end;
          except
            Progress.Log(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
          end;
        end;

      finally
        LineList.free;
      end;
    end;

    // Row Count the matched table
    if sql_tbl_count > 0 then
    begin
      sql_row_count := 0;
      sqlselect_row := TSQLite3Statement.Create(mydb, 'SELECT COUNT(*) FROM ' + gArr[refnum].fi_SQLPrimary_Tablestr);
      try
        while sqlselect_row.Step = SQLITE_ROW do
        begin
          sql_row_count := sqlselect_row.ColumnInt(0);
          break;
        end;
      finally
        sqlselect_row.free;
      end;
    end;

  end;

  // Process As ESE ---------------------------------------
  procedure ProcessAsESE;
  var
    anArray: array of integer;
    ContainerID_name_str: string;
    EntryReader: TEntryReader;
    ese_col: string;
    eseQuery: TESEQuery;
    FESE: TESE; // called per edb entry
    FESEDatabase: TESEDataBase;
    Tbl: TeseTable;
    j, k: integer;

    function ColumnAsDT(col: TEseColumn): TDateTime;
    var
      temp_int64: int64;
    begin
      Result := 0;
      if assigned(col) then
        if col.ColType = ctLongLong then
        begin
          temp_int64 := col.ReadAsUInt64;
          try
            Result := Int64ToDateTime_ConvertAs(temp_int64 div 10, 'DTChrome'); // KVL - Force the conversion type.
          except
            on e: exception do
            begin
              Progress.Log(e.message);
              Progress.Log(HYPHEN + 'Int64ToDateTime_ConvertAs');
            end;
          end;
        end;
    end;

    function ColumnToInteger(col: TEseColumn): int64;
    begin
      Result := 0;
      if assigned(col) then
        case col.ColType of
          ctBoolean:
            if col.ReadAsBoolean then
              Result := 1
            else
              Result := 0;
          ctByte:
            Result := col.ReadAsbyte;
          ctShort:
            Result := col.ReadAsWord;
          ctLong:
            Result := col.ReadAsinteger;
          ctDatetime:
            Result := col.ReadAsint64;
          ctFloatSingle, ctFloatDouble, ctCurrency, ctBinary, ctLongBinary, ctSLV, ctGUID:
            Result := col.ReadAsint64;
          ctUnsignedLong:
            Result := col.ReadAsDWord;
          ctLongLong:
            Result := col.ReadAsint64;
          ctUnsignedShort:
            Result := col.ReadAsWord;
        end;
    end;

    function ColumnToString(col: TEseColumn): string;
    begin
      Result := '';
      if assigned(col) then
        case col.ColType of
          ctBoolean:
            if col.ReadAsBoolean then
              Result := 'True'
            else
              Result := 'False';
          ctByte:
            Result := col.ReadAsHexString;
          ctShort:
            Result := IntToStr(col.ReadAsWord);
          ctLong:
            Result := IntToStr(col.ReadAsinteger);
          ctDatetime:
            Result := DateTimeToStr(col.ReadAsDateTime);
          ctFloatSingle, ctFloatDouble, ctCurrency, ctBinary, ctLongBinary, ctSLV, ctGUID:
            Result := col.ReadAsHexString;
          ctText, ctLongText:
            Result := col.ReadAsString;
          ctUnsignedLong:
            Result := IntToStr(col.ReadAsDWord);
          ctLongLong:
            Result := IntToStr(col.ReadAsUInt64);
          ctUnsignedShort:
            Result := IntToStr(col.ReadAsWord);
          ctMaxColType:
            Result := 'MaxColType';
        end;
    end;

    procedure ReadESEFields;
    var
      hex_char, hex_str: string;
      t, page_title_length, start_byte, end_byte: integer;
      BytesArray: Tbytes;
      iii: integer;
      new_str: string;
      ese_col: string;
    begin
      for g := 1 to ColCount do
      begin
        if not Progress.isRunning then
          break;
        DNT_sql_col := aItems[g].sql_col;
        ese_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
        if (DNT_sql_col = DNT_AccessedTime) then
          variant_Array[g] := ColumnAsDT(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_ContainerID) then
          variant_Array[g] := ColumnToInteger(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_CreationTime) then
          variant_Array[g] := ColumnAsDT(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_EntryID) then
          variant_Array[g] := ColumnToInteger(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_ExpiryTime) then
          variant_Array[g] := ColumnAsDT(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_Filename) then
          variant_Array[g] := ColumnToString(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_Filesize) then
          variant_Array[g] := ColumnToInteger(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_Flags) then
          variant_Array[g] := ColumnToString(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_AccessCount) then
          variant_Array[g] := ColumnToInteger(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_ModifiedTime) then
          variant_Array[g] := ColumnAsDT(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_RecoveryType) then
          variant_Array[g] := 'ESE Parse';
        if (DNT_sql_col = DNT_SyncTime) then
          variant_Array[g] := ColumnAsDT(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_Url) then
          variant_Array[g] := ColumnToString(eseQuery.CurrentRow.ColumnByName(ese_col));
        if (DNT_sql_col = DNT_UrlHash) then
          variant_Array[g] := ColumnToString(eseQuery.CurrentRow.ColumnByName(ese_col));

        // ESE History Response Header for Page Title
        if (Item.fi_ESE_ContainerName = RS_DNT_History) then
        begin
          if (DNT_sql_col = DNT_ResponseHeaders) then
          begin
            if assigned(eseQuery.CurrentRow.ColumnByName(ese_col)) then
            begin
              BytesArray := eseQuery.CurrentRow.ColumnByName(ese_col).ReadAsTBytes;
              if Length(BytesArray) > 64 then
              begin
                for b := 0 to Length(BytesArray) - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  hex_str := '';
                  new_str := '';
                  hex_char := '';
                  page_title_length := 0;
                  // Create a string of array values in hex
                  hex_char := inttohex(BytesArray[b], 2);
                  hex_str := hex_str + hex_char;
                  // Test to locate start of page title after 1F-00-00-00
                  if hex_char = '1F' then
                  begin
                    hex_char := hex_char + '-' + inttohex(BytesArray[b + 1], 2);
                    hex_char := hex_char + '-' + inttohex(BytesArray[b + 2], 2);
                    hex_char := hex_char + '-' + inttohex(BytesArray[b + 3], 2);
                    // If 1F-00-00-00 found set the start byte
                    if hex_char = '1F-00-00-00' then
                    begin
                      start_byte := b + 8;
                      hex_str := '';
                      hex_char := '';
                      // Read from the start byte
                      for iii := start_byte to high(BytesArray) do
                      begin
                        if not Progress.isRunning then
                          break;
                        hex_char := inttohex(BytesArray[iii], 2);
                        hex_str := hex_str + hex_char;
                        // Test look for the terminator 00-00
                        if hex_char = '00' then
                        begin
                          hex_char := hex_char + '-' + inttohex(BytesArray[iii + 1], 2);
                          // If terminator found set the end byte and Page Title length
                          if hex_char = '00-00' then
                          begin
                            end_byte := iii;
                            page_title_length := end_byte - start_byte;
                            // Read the Page Title into a char string
                            if page_title_length > 0 then
                            begin
                              for t := start_byte to end_byte do
                                if BytesArray[t] <> 0 then
                                  new_str := new_str + chr(BytesArray[t]);
                              if VERBOSE then
                                Progress.Log('ESE History Page Title: Start Byte: ' + IntToStr(start_byte) + SPACE + 'End Byte: ' + IntToStr(end_byte) + SPACE + 'length: ' + IntToStr(page_title_length) + SPACE + new_str);
                              variant_Array[g] := new_str;
                              break; // No further processing of the response header
                            end;
                          end;
                        end;
                      end;
                      break;
                    end;
                  end; { search for 1F marker }
                end; { read full byte array }
              end;
              variant_Array[g] := new_str;
            end;
          end;
        end;

        if (Item.fi_ESE_ContainerName = RS_DNT_iedownload) then
          if (DNT_sql_col = DNT_ResponseHeaders) and assigned(eseQuery.CurrentRow.ColumnByName(ese_col)) then
          begin
            BytesArray := eseQuery.CurrentRow.ColumnByName(ese_col).ReadAsTBytes;
            new_str := '';
            // Read backwards form the end of the response header to get the file name after the first found terminator
            for iii := high(BytesArray) - 2 downto 1 do
            begin
              hex_char := inttohex(BytesArray[iii], 2);
              hex_str := hex_str + hex_char;
              // Test look for the terminator 00-00
              if hex_char = '00' then
              begin
                hex_char := hex_char + '-' + inttohex(BytesArray[iii - 1], 2);
                if hex_char = '00-00' then
                begin
                  start_byte := iii;
                  // Read the download file into a char string
                  for t := start_byte to high(BytesArray) do
                    if BytesArray[t] <> 0 then
                      new_str := new_str + chr(BytesArray[t]);
                  if VERBOSE then
                    Progress.Log('ESE History Page Title: Start Byte: ' + IntToStr(start_byte) + SPACE + 'End Byte: ' + IntToStr(end_byte) + SPACE + 'length: ' + IntToStr(page_title_length) + SPACE + new_str);
                  variant_Array[g] := new_str;
                  break; // No further processing of the response header
                end;
              end;
            end;
            variant_Array[g] := new_str;
          end; { ieDownload }
      end;
    end;

  // Start of Process As ESE ---------------------------------------------------
  begin
    NullTheArray;

    // Open the Data Stream;
    EntryReader := TEntryReader.Create;
    EntryReader.OpenData(aArtifactEntry);
    try
      // Create reader and open
      FESE := TESE.Create(Progress); // PacProgress can be nil;
      FESEDatabase := TESEDataBase.Create(FESE);
      try
        // Open ESE Stream
        if VERBOSE then Progress.Log('Open ESE stream');
        FESE.open(EntryReader, nil, eomQuickTable);
        FESEDatabase.OpenStream(EntryReader, nil);

        if VERBOSE then Progress.Log(StringOfChar('-', CHAR_LENGTH));

        // Find table
        if VERBOSE then Progress.Log(RPad('Total Tables:', RPAD_VALUE) + IntToStr(FESE.Tables.Count));

        Tbl := nil;
        for Tbl in FESE.Tables do
        begin
          if not Progress.isRunning then
            break;

          if (Tbl.TableName = 'Containers') then
          begin
            // Open the table found
            if VERBOSE then Progress.Log(StringOfChar('-', CHAR_LENGTH));

            eseQuery := FESEDatabase.OpenTable(Tbl);
            if assigned(eseQuery) then
              try
                if VERBOSE then Progress.Log(RPad('Opened Table:', RPAD_VALUE) + Tbl.TableName);
                Progress.DisplayMessages := ('Opened Table:' + SPACE + Tbl.TableName);

                if VERBOSE then Progress.Log(RPad('Unique "Name" col values:', RPAD_VALUE) + IntToStr(Tbl.Coldefslist.Count));
                ContainerID_name_str := Item.fi_ESE_ContainerName;

                if VERBOSE then Progress.Log(RPad('Search ' + Tbl.TableName + '\Name for:', RPAD_VALUE) + '"' + ContainerID_name_str + '"');

                eseQuery.First;
                setlength(anArray, Tbl.RecordCount);
                k := 0;

                // Loop over 'containers' find all rows with "content" in name column and get the ContainerID from that row and store into array
                while not eseQuery.EOF and Progress.isRunning do
                begin
                  // Get the contents of name column
                  if eseQuery.CurrentRow.ColumnByName(RS_DNT_Name1).ReadAsString = ContainerID_name_str then // name
                  begin
                    // Add item to array
                    if VERBOSE then Progress.Log(RPad('Found "' + eseQuery.CurrentRow.ColumnByName(RS_DNT_Name1).ReadAsString + '"', RPAD_VALUE) + 'ContainerId' + SPACE + IntToStr(eseQuery.CurrentRow.ColumnByName('ContainerId').ReadAsinteger));
                    anArray[k] := ColumnToInteger(eseQuery.CurrentRow.ColumnByName('ContainerId')); // ContainerID
                    Inc(k);
                  end;
                  eseQuery.Next;
                end; { End while }
                setlength(anArray, k);
                break;
              finally
                eseQuery.free;
              end;
          end;
        end;

        if VERBOSE then Progress.Log(RPad('Added tables to anArray:', RPAD_VALUE) + IntToStr(high(anArray)));
        if VERBOSE then Progress.Log(StringOfChar('-', CHAR_LENGTH));

        // Process IE 10-11 ESE Database Tables (Dependency is processed separately below)
        if not(Item.fi_ESE_ContainerName = 'DependencyEntryEx') then // !NOT
        begin
          for Tbl in FESE.Tables do
          begin
            if not Progress.isRunning then break;

            for j := 0 to high(anArray) do
            begin
              if not Progress.isRunning then
                break;

              if VERBOSE then Progress.Log('Testing ' + Tbl.TableName + ' =' + 'Container_' + IntToStr(anArray[j]));
              if (Tbl.TableName = 'Container_' + IntToStr(anArray[j])) then
              begin

                if VERBOSE then Progress.Log(RPad('Processing Table:', RPAD_VALUE) + Tbl.TableName);
                // Open the table found

                // Tbl := EseReader.Tablelist[m]; // Check we have a table and that it is opened
                eseQuery := FESEDatabase.OpenTable(Tbl);
                if assigned(eseQuery) then
                  try
                    if VERBOSE then Progress.Log(RPad('Opened:', RPAD_VALUE) + Tbl.TableName);
                    if VERBOSE then
                      Progress.DisplayMessages := 'Opened' + SPACE + Tbl.TableName;
                    eseQuery.First;
                    k := 0;
                    g := 1;

                    while not eseQuery.EOF and Progress.isRunning do
                    begin
                      NullTheArray;
                      if (Item.fi_ESE_ContainerName = RS_DNT_Content) then
                        ReadESEFields;
                      if (Item.fi_ESE_ContainerName = RS_DNT_Cookies) then
                        ReadESEFields;
                      if (Item.fi_ESE_ContainerName = RS_DNT_History) then
                        ReadESEFields;
                      if (Item.fi_ESE_ContainerName = RS_DNT_iedownload) then
                        ReadESEFields;
                      AddToModule;
                      eseQuery.Next;
                      Inc(k);
                    end; { while }

                    Progress.DisplayMessages := 'Read rows' + SPACE + Tbl.TableName + RUNNING;
                    if VERBOSE then Progress.Log(RPad('Total rows:', RPAD_VALUE) + IntToStr(k));
                  finally
                    eseQuery.free;
                  end
                else if VERBOSE then Progress.Log('Table not found or Not Opened');

                if VERBOSE then Progress.Log(StringOfChar('-', CHAR_LENGTH));
              end;
            end;
          end;
        end;

        // Process IE 10-11 ESE Database Tables (Dependency Only)
        if (Item.fi_ESE_ContainerName = 'DependencyEntryEx') then
          for Tbl in FESE.Tables do
          begin
            if not Progress.isRunning then
              break;

            if RegexMatch(Tbl.TableName, RS_DNT_Dependency, False) then
            begin
              eseQuery := FESEDatabase.OpenTable(Tbl);
              if assigned(eseQuery) then
                try
                  eseQuery.First;
                  k := 0;
                  g := 1;
                  while not eseQuery.EOF and Progress.isRunning do
                  begin
                    NullTheArray;
                    // The Discrepancy table has mostly unique fields
                    for g := 1 to ColCount do
                    begin
                      DNT_sql_col := aItems[g].sql_col;
                      ese_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));

                      if (DNT_sql_col = DNT_EntryID) then
                        variant_Array[g] := ColumnToInteger(eseQuery.CurrentRow.ColumnByName(ese_col));

                      if (DNT_sql_col = DNT_Hostname) then
                        variant_Array[g] := ColumnToString(eseQuery.CurrentRow.ColumnByName(ese_col));

                      if (DNT_sql_col = DNT_ModifiedTime) then
                        variant_Array[g] := ColumnAsDT(eseQuery.CurrentRow.ColumnByName(ese_col));

                      if (DNT_sql_col = DNT_Port) then
                        variant_Array[g] := ColumnToString(eseQuery.CurrentRow.ColumnByName(ese_col));

                      if (DNT_sql_col = DNT_RecoveryType) then
                        variant_Array[g] := 'ESE Parse';

                      if (DNT_sql_col = DNT_Url) then
                        variant_Array[g] := ColumnToString(eseQuery.CurrentRow.ColumnByName(ese_col));

                      if (DNT_sql_col = DNT_UrlSchemaType) then
                        variant_Array[g] := ColumnToString(eseQuery.CurrentRow.ColumnByName(ese_col));
                    end;
                    AddToModule;
                    eseQuery.Next;
                    Inc(k);
                  end; { end of table }
                finally
                  eseQuery.free;
                end;
            end; { if table name }
          end { for table list };
      finally
        FreeAndNil(FESE);
      end;
    finally
      FreeAndNil(EntryReader);
    end;
  end; { Procedure: Process as ESE Table }

  function CarveAccessCount(hdr_start_of_regex_match: int64): variant;
  var
    AccessCount_int, AccessCount_offset_end, AccessCount_offset_start: int64;
  begin
    Result := null;
    HeaderReader.Position := hdr_start_of_regex_match - 62;
    AccessCount_offset_start := HeaderReader.Position;
    AccessCount_int := HeaderReader.AsInteger;
    AccessCount_offset_end := HeaderReader.Position;
    if VERBOSE then Progress.Log(RPad(DNT_sql_col + COLON, RPAD_VALUE) + IntToStr(AccessCount_int) + SPACE + '(Bytes read: ' + IntToStr(AccessCount_offset_end - AccessCount_offset_start) + ', starting at: ' + IntToStr(AccessCount_offset_start) + ')');
    if (AccessCount_int > 0) and (AccessCount_int < 10000) then // Sanity check
      Result := AccessCount_int;
  end;

  function SetFileOffset(hdr_start_of_regex_match: int64): variant;
  begin
    Result := null;
    if VERBOSE then Progress.Log(RPad(DNT_sql_col + COLON, RPAD_VALUE) + IntToStr(hdr_start_of_regex_match));
    Result := hdr_start_of_regex_match;
  end;

  function CarveFlags(hdr_start_of_regex_match: int64): variant;
  var
    Flags_int, Flags_offset_end, Flags_offset_start: int64;
    Flags_str: string;
  begin
    Result := null;
    Flags_str := '';
    HeaderReader.Position := hdr_start_of_regex_match - 66;
    Flags_offset_start := HeaderReader.Position;
    Flags_int := HeaderReader.AsInteger;
    Flags_offset_end := HeaderReader.Position;
    if VERBOSE then
      Progress.Log(RPad(DNT_sql_col + COLON, RPAD_VALUE) + IntToStr(Flags_int) + SPACE + '(Bytes read: ' + IntToStr(Flags_offset_end - Flags_offset_start) + ', starting at: ' + IntToStr(Flags_offset_start) + ')');
    if (Flags_int > 0) and (Flags_int < 200) then // Sanity check
    begin
      Flags_str := IntToStr(Flags_int);
      Result := Flags_str;
    end;
  end;

  function SetRecoveryType(const AString: string): variant;
  begin
    if VERBOSE then
      Progress.Log(RPad(DNT_sql_col + COLON, RPAD_VALUE) + AString);
    Result := AString;
  end;

  function CarveURL(carve_position: int64): variant;
  var
    printchars: string;
  begin
    HeaderReader.Position := carve_position;
    printchars := HeaderReader.AsNullWideString; // Stops printing when it reaches a null char
    printchars := StringReplace(printchars, '%20', ' ', [rfReplaceAll]);
    // Progress.Log(printchars); // Prints the carved URL
    if VERBOSE then Progress.Log(RPad(DNT_sql_col + COLON, RPAD_VALUE) + printchars);
    Result := printchars;
  end;

  function CarveURLHash(carve_position: int64): variant;
  var
    urlhash_int: int64;
    urlhash_str: string;
  begin
    urlhash_str := '';
    HeaderReader.Position := carve_position - 90;
    urlhash_int := HeaderReader.Asint64; // Stops printing when it reaches a null char
    if (Length(urlhash_int) >= 17) and (Length(urlhash_int) <= 19) then
      urlhash_str := IntToStr(urlhash_int);
    if VERBOSE then Progress.Log(RPad(DNT_sql_col + COLON, RPAD_VALUE) + urlhash_str);
    Result := urlhash_str;
  end;

  function CarveFileName(carve_position: int64): variant;
  var
    printchars: string;
  begin
    HeaderReader.Position := carve_position;
    HeaderReader.AsNullWideString; // Takes header reader to the end of the url
    HeaderReader.Position := HeaderReader.Position + 1;
    printchars := HeaderReader.AsNullWideString;
    printchars := StringReplace(printchars, '%20', ' ', [rfReplaceAll]);
    if VERBOSE then Progress.Log(RPad(DNT_sql_col + COLON, RPAD_VALUE) + printchars);
    Result := printchars;
  end;

  function CarveFileSize(carve_position: int64): variant;
  var
    hex_str, filesize_str: string;
    FileSizeStart_int64, FileSizeEnd_int64, distance_to_terminator: int64;
    m, ndx, FileSize_int, FileSizeBytes_int: integer;
  begin
    Result := null;
    filesize_str := '';
    FileSize_int := 0;
    HeaderReader.Position := carve_position;
    HeaderReader.AsNullWideString; // Takes header reader to the end of the url
    HeaderReader.Position := HeaderReader.Position + 1; // Takes the header reader to the end of the file name
    HeaderReader.AsNullWideString;
    HeaderReader.Position := HeaderReader.Position + 1;
    for m := 1 to 200 do
    begin
      if not Progress.isRunning then
        break;

      hex_str := HeaderReader.AsHexString(16);
      if HeaderReader.Position >= HeaderReader.Size then
      begin
        MessageUser('Error. HeaderReader exceeded HeaderReader size.');
        Exit;
      end;

      if (hex_str = ' 43 6F 6E 74 65 6E 74 2D 4C 65 6E 67 74 68 3A 20') then // "Content-Length: "
      begin
        FileSizeStart_int64 := HeaderReader.Position;

        // Locate the terminator immediately after the file size
        distance_to_terminator := 20;
        for ndx := 0 to distance_to_terminator - 1 do
        begin
          if not Progress.isRunning then
            break;
          hex_str := HeaderReader.AsHexString(2);

          if HeaderReader.Position >= HeaderReader.Size then
          begin
            MessageUser('Error. HeaderReader exceeded HeaderReader size.');
            Exit;
          end;

          if hex_str = ' 0D 0A' then
          begin
            FileSizeEnd_int64 := HeaderReader.Position - 2;
            FileSizeBytes_int := FileSizeEnd_int64 - FileSizeStart_int64;
            HeaderReader.Position := FileSizeStart_int64;
            filesize_str := HeaderReader.AsPrintableChar(FileSizeBytes_int);
            if VERBOSE then Progress.Log(RPad('File Size:', RPAD_VALUE) + filesize_str + SPACE + '(start: ' + IntToStr(FileSizeStart_int64) + SPACE + 'end: ' + IntToStr(FileSizeEnd_int64) + ')');

            try
              FileSize_int := StrToInt(filesize_str);
              Result := FileSize_int;
            except
              Progress.Log(ATRY_EXCEPT_STR + 'Error converting file size from string.');
            end;

            if VERBOSE then Progress.Log(RPad(DNT_sql_col + COLON, RPAD_VALUE) + IntToStr(FileSize_int));
            break;
          end;
          HeaderReader.Position := HeaderReader.Position - 1;
        end;
        if ndx = distance_to_terminator then
          Progress.Log('Did not find terminator.');
        Exit; // Exit if not terminator found
      end;
      HeaderReader.Position := HeaderReader.Position - 15;
    end;
  end;

  // Print 5 dates when in verbose mode ----------------------------------------
  procedure VerboseDate(start_carve_position: int64);
  var
    dt_array: array of TDateTime;
    aNewDateTime: TDateTime;
    aNewDate_int, date_start_int, date_end_int: int64;
    p: integer;
    msg_str, date_str: string;
  begin
    if VERBOSE then
    begin
      setlength(dt_array, 6);
      HeaderReader.Position := start_carve_position - 58; // -18 -8 -8 -8 -8 -8
      Progress.Log((format('%-29s %-20s %-20s %-20s %-20s %-20s', ['Dates Found', 'Start', 'End', 'Total bytes', 'Date Raw', 'Date'])));
      for p := 5 downto 1 do
      begin
        aNewDate_int := 0;
        aNewDateTime := null;
        date_str := '';
        try
          date_start_int := HeaderReader.Position;
          aNewDate_int := HeaderReader.Asint64; // Header reader moves 8 bytes forward
          date_end_int := HeaderReader.Position - date_start_int;
          if DateCheck_Unix_10(aNewDate_int) then
          begin
            aNewDateTime := FileTimeToDateTime(aNewDate_int);
            dt_array[p] := aNewDateTime;
            date_str := DateTimeToStr(aNewDateTime);
          end;
          if p = 5 then
            msg_str := 'Sync Time';
          if p = 4 then
            msg_str := 'Creation Time';
          if p = 3 then
            msg_str := 'Expiry Time';
          if p = 2 then
            msg_str := 'Modified Time';
          if p = 1 then
            msg_str := 'Accessed Time';
          Progress.Log((format('%-29s %-20s %-20s %-20s %-20s %-20s', [IntToStr(p) + ': ' + msg_str, IntToStr(date_start_int), IntToStr(HeaderReader.Position), IntToStr(HeaderReader.Position - date_start_int), IntToStr(aNewDate_int),
            date_str])));
        except
          MessageUser('Error reading date.');
          Exit;
        end;
      end;
      Progress.Log(' ');
    end;
  end;

// Function: Carve Date ------------------------------------------------------
// From \xFE.. backwards: (Regex match starts here)
// - 18 back to end of first date  = 18
// - 08 Access Time                = 26
// - 08 Modified Time              = 34
// - 08 Expiry Time                = 42
// - 08 Creation Time              = 50
// - 08 Sync Time                  = 58
// - 04 start of access count      = 62
// - 04 flag                       = 66
// - 24 URLHash                    = 90
// - total

  function CarveDate(start_carve_position: int64; DateToUse_int: integer): variant;
  var
    aNewDateTime: TDateTime;
    aNewDate_int: int64;
    GetDateAtPosition: int64;
  begin
    Result := null;
    if (DateToUse_int >= 1) and (DateToUse_int <= 5) then
    begin
      GetDateAtPosition := start_carve_position - 18 - (DateToUse_int * 8);
      HeaderReader.Position := GetDateAtPosition;
      try
        aNewDate_int := HeaderReader.Asint64; // Header reader moves 8 bytes forward
        if DateCheck_Unix_10(aNewDate_int) then
        begin
          aNewDateTime := FileTimeToDateTime(aNewDate_int);
          Result := aNewDateTime;
        end;
      except
        MessageUser('Error reading date.');
      end;
    end;
  end;

  procedure PrintCarveHeader(hdr_start: int64; hdr_end: int64; hdr_count: int64);
  begin
    if VERBOSE then
      Progress.Log((format('%-29s %-20s %-20s %-20s', ['Regex Match: Visited', 'Start', 'End', 'Total bytes'])));
    if VERBOSE then
      Progress.Log((format('%-29s %-20s %-20s %-20s', ['', IntToStr(hdr_start), IntToStr(hdr_end), IntToStr(hdr_count)])));
    if VERBOSE then
      Progress.Log(' ');
  end;

  function CarveReference(hdr_end: int64): string;
  var
    printchars: string;
  begin
    Result := '';
    HeaderReader.Position := hdr_end - 36;
    try
      printchars := HeaderReader.AsPrintableChar(32);
    except
      MessageUser('Error reading carve of "Reference".');
      Exit;
    end;
    printchars := StringReplace(printchars, '.', '', [rfReplaceAll]);
    Result := printchars;
    if VERBOSE then
      Progress.Log(RPad(DNT_sql_col + COLON, RPAD_VALUE) + Result);
  end;

  procedure ProcessAsESECarveContent;
  var
    hdr_start_of_regex_match: int64;
    hdr_count_of_matched_bytes: int64;
    hdr_last_matched_char: int64;
  begin
    if (UpperCase(Item.fi_Process_As) = PROCESS_AS_ESE_CARVE_CONTENT) then
    begin
      if HeaderReader.OpenData(aArtifactEntry) then // Open the entry to search
      begin
        try
          HeaderReader.Position := 0;
          HeaderRegex.Stream := HeaderReader;
          HeaderRegex.Find(hdr_start_of_regex_match, hdr_count_of_matched_bytes);
          while (hdr_start_of_regex_match <> -1) do
          begin
            if not Progress.isRunning then
              break;
            NullTheArray;
            curpos := HeaderReader.Position; // Save the position

            // Set starting position (do not go past the beginning of the file)
            if hdr_start_of_regex_match < 0 then
              hdr_start_of_regex_match := 0;

            // Set ending position (do not go past the end of the file)
            hdr_last_matched_char := hdr_start_of_regex_match + hdr_count_of_matched_bytes;
            if hdr_last_matched_char >= HeaderReader.Size then
              hdr_last_matched_char := HeaderReader.Size - 1;

            if VERBOSE then
              PrintCarveHeader(hdr_start_of_regex_match, hdr_last_matched_char, hdr_count_of_matched_bytes);
            if VERBOSE then
              VerboseDate(hdr_start_of_regex_match);

            // Put in this loop so that the order of columns positions can be easily changed
            for g := 1 to ColCount do
            begin
              DNT_sql_col := aItems[g].sql_col;
              if not Progress.isRunning then
                break;
              if (DNT_sql_col = DNT_AccessedTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 1); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_ModifiedTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 2); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_ExpiryTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 3); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_CreationTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 4); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_SyncTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 5); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_AccessCount) then
                variant_Array[g] := CarveAccessCount(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_Filename) then
                variant_Array[g] := CarveFileName(hdr_last_matched_char - 8); // -8 is back from http
              if (DNT_sql_col = DNT_FileOffset) then
                variant_Array[g] := SetFileOffset(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_Filesize) then
                variant_Array[g] := CarveFileSize(hdr_last_matched_char);
              if (DNT_sql_col = DNT_Flags) then
                variant_Array[g] := CarveFlags(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_RecoveryType) then
                variant_Array[g] := SetRecoveryType(Item.fi_Recovery_Type);
              if (DNT_sql_col = DNT_Url) then
                variant_Array[g] := CarveURL(hdr_last_matched_char - 8);
              if (DNT_sql_col = DNT_UrlHash) then
                variant_Array[g] := CarveURLHash(hdr_start_of_regex_match);
            end;
            AddToModule;
            if VERBOSE then
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

            // Restore the position and find the next match
            HeaderReader.Position := curpos;
            HeaderRegex.FindNext(hdr_start_of_regex_match, hdr_count_of_matched_bytes);
          end;
        except
          MessageUser('Error processing ' + aArtifactEntry.EntryName);
        end;
      end;
    end;
  end;

  procedure ProcessAsESECarveCookies;
  var
    hdr_start_of_regex_match: int64;
    hdr_count_of_matched_bytes: int64;
    hdr_last_matched_char: int64;
  begin
    if (UpperCase(Item.fi_Process_As) = PROCESS_AS_ESE_CARVE_COOKIES) then
    begin
      //Progress.Log(StringOfChar('B', 100) + SPACE + Item.fi_Process_ID + ' >>> ' + Item.fi_Process_As + ' >>> ' + Item.fi_ESE_ContainerName);

      if HeaderReader.OpenData(aArtifactEntry) then // Open the entry to search
      begin
        try
          HeaderReader.Position := 0;
          HeaderRegex.Stream := HeaderReader;
          HeaderRegex.Find(hdr_start_of_regex_match, hdr_count_of_matched_bytes);
          while (hdr_start_of_regex_match <> -1) do
          begin
            if not Progress.isRunning then
              break;
            NullTheArray;

            // Save the position
            curpos := HeaderReader.Position;

            // Set starting position (do not go past the beginning of the file)
            if hdr_start_of_regex_match < 0 then
              hdr_start_of_regex_match := 0;

            // Set ending position (do not go past the end of the file)
            hdr_last_matched_char := hdr_start_of_regex_match + hdr_count_of_matched_bytes;
            if hdr_last_matched_char >= HeaderReader.Size then
              hdr_last_matched_char := HeaderReader.Size - 1;

            if VERBOSE then
              PrintCarveHeader(hdr_start_of_regex_match, hdr_last_matched_char, hdr_count_of_matched_bytes);
            if VERBOSE then
              VerboseDate(hdr_start_of_regex_match);

            // Put in this loop so that the order of columns positions can be easily changed
            for g := 1 to ColCount do
            begin
              DNT_sql_col := aItems[g].sql_col;
              if not Progress.isRunning then
                break;
              if (DNT_sql_col = DNT_AccessedTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 1); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_ModifiedTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 2); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_ExpiryTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 3); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_CreationTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 4); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_SyncTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 5); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_AccessCount) then
                variant_Array[g] := CarveAccessCount(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_Filename) then
                variant_Array[g] := CarveFileName(hdr_last_matched_char - 8); // -8 is back from http
              if (DNT_sql_col = DNT_FileOffset) then
                variant_Array[g] := SetFileOffset(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_Flags) then
                variant_Array[g] := CarveFlags(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_RecoveryType) then
                variant_Array[g] := SetRecoveryType(Item.fi_Recovery_Type);
              if (DNT_sql_col = DNT_Url) then
                variant_Array[g] := CarveURL(hdr_last_matched_char - 14);
              if (DNT_sql_col = DNT_UrlHash) then
                variant_Array[g] := CarveURLHash(hdr_start_of_regex_match);
            end;
            AddToModule;
            if VERBOSE then
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

            // Restore the position and find the next match
            HeaderReader.Position := curpos;
            HeaderRegex.FindNext(hdr_start_of_regex_match, hdr_count_of_matched_bytes);
          end;
        except
          MessageUser('Error processing ' + aArtifactEntry.EntryName);
        end;
      end;
    end;
  end;

  procedure ProcessAsESECarveHistoryVisited;
  var
    hdr_start_of_regex_match: int64;
    hdr_count_of_matched_bytes: int64;
    hdr_last_matched_char: int64;
    temp_pos_str: string;
    printchars: string;
    ndx: integer;
    hex_str: string;

  begin
  if (UpperCase(Item.fi_Process_As) = PROCESS_AS_ESE_CARVE_HISTORY_VISITED) then
    begin
      //Progress.Log(StringOfChar('C', 100) + SPACE + Item.fi_Process_ID + ' >>> ' + Item.fi_Process_As + ' >>> ' + Item.fi_ESE_ContainerName);

      if HeaderReader.OpenData(aArtifactEntry) then // Open the entry to search
      begin
        try
          HeaderReader.Position := 0;
          HeaderRegex.Stream := HeaderReader;
          HeaderRegex.Find(hdr_start_of_regex_match, hdr_count_of_matched_bytes);
          while (hdr_start_of_regex_match <> -1) do
          begin
            if not Progress.isRunning then
              break;
            NullTheArray;

            // Save the position
            curpos := HeaderReader.Position;

            // Set starting position (do not go past the beginning of the file)
            if hdr_start_of_regex_match < 0 then
              hdr_start_of_regex_match := 0;

            // Set ending position (do not go past the end of the file)
            hdr_last_matched_char := hdr_start_of_regex_match + hdr_count_of_matched_bytes;
            if hdr_last_matched_char >= HeaderReader.Size then
              hdr_last_matched_char := HeaderReader.Size - 1;

            if VERBOSE then
              PrintCarveHeader(hdr_start_of_regex_match, hdr_last_matched_char, hdr_count_of_matched_bytes);
            if VERBOSE then
              VerboseDate(hdr_start_of_regex_match);

            // Put in this loop so that the order of columns positions can be easily changed
            for g := 1 to ColCount do
            begin
              DNT_sql_col := aItems[g].sql_col;
              if not Progress.isRunning then
                break;
              printchars := '';

              if (DNT_sql_col = DNT_AccessedTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 1); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_ModifiedTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 2); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_ExpiryTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 3); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_CreationTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 4); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_SyncTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 5); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_AccessCount) then
                variant_Array[g] := CarveAccessCount(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_FileOffset) then
                variant_Array[g] := SetFileOffset(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_Flags) then
                variant_Array[g] := CarveFlags(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_RecoveryType) then
                variant_Array[g] := SetRecoveryType(Item.fi_Recovery_Type);
              if (DNT_sql_col = DNT_Url) then
                variant_Array[g] := CarveURL(hdr_last_matched_char - 18); // The URL string is to include "Visited:"
              if (DNT_sql_col = DNT_UrlHash) then
                variant_Array[g] := CarveURLHash(hdr_start_of_regex_match);

              // Carve_History_Visited - Page Title ------------------------------
              if (DNT_sql_col = DNT_PageTitle) then
              begin
                // Set the starting position at the terminator at the end of the URL
                HeaderReader.Position := hdr_start_of_regex_match;
                temp_pos_str := HeaderReader.AsNullWideString; // Stops printing when it reaches a null char
                HeaderReader.Position := HeaderReader.Position + 1; // Takes 1 forward to the 01 of 00 00 00 01.
                // Locate and read the page title
                HeaderReader.Position := HeaderReader.Position + 8;
                printchars := HeaderReader.AsPrintableChar(4);
                if printchars = '1SPS' then
                begin
                  for ndx := 1 to 100 do
                  begin
                    hex_str := HeaderReader.AsHexString(4);
                    if (hex_str = ' 1F 00 00 00') then
                    begin
                      HeaderReader.Position := HeaderReader.Position + 4;
                      printchars := HeaderReader.AsNullWideString;
                      if Length(printchars) > 4 then
                        variant_Array[g] := printchars;
                      break;
                    end;
                    HeaderReader.Position := HeaderReader.Position - 3;
                  end;
                end;
              end; { Page Title }

            end;
            AddToModule;
            if VERBOSE then
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

            // Restore the position and find the next match
            HeaderReader.Position := curpos;
            HeaderRegex.FindNext(hdr_start_of_regex_match, hdr_count_of_matched_bytes);
          end;
        except
          MessageUser('Error processing ' + aArtifactEntry.EntryName);
        end;
      end;
    end;
  end;

  procedure ProcessAsESECarveHistoryRefMarker;
  var
    hdr_start_of_regex_match: int64;
    hdr_count_of_matched_bytes: int64;
    hdr_last_matched_char: int64;
  begin
    if (UpperCase(Item.fi_Process_As) = PROCESS_AS_ESE_CARVE_HISTORY_REF_MARKER) then
    begin
      //Progress.Log(StringOfChar('D', 100) + SPACE + Item.fi_Process_ID + ' >>> ' + Item.fi_Process_As + ' >>> ' + Item.fi_ESE_ContainerName);

      if HeaderReader.OpenData(aArtifactEntry) then // Open the entry to search
      begin
        try
          HeaderReader.Position := 0;
          HeaderRegex.Stream := HeaderReader;
          HeaderRegex.Find(hdr_start_of_regex_match, hdr_count_of_matched_bytes);
          while (hdr_start_of_regex_match <> -1) do
          begin
            if not Progress.isRunning then
              break;
            NullTheArray;

            // Save the position
            curpos := HeaderReader.Position;

            // Set starting position (do not go past the beginning of the file)
            if hdr_start_of_regex_match < 0 then
              hdr_start_of_regex_match := 0;

            // Set ending position (do not go past the end of the file)
            hdr_last_matched_char := hdr_start_of_regex_match + hdr_count_of_matched_bytes;
            if hdr_last_matched_char >= HeaderReader.Size then
              hdr_last_matched_char := HeaderReader.Size - 1;

            if VERBOSE then
              PrintCarveHeader(hdr_start_of_regex_match, hdr_last_matched_char, hdr_count_of_matched_bytes);
            if VERBOSE then
              VerboseDate(hdr_start_of_regex_match);

            // Put in this loop so that the order of columns positions can be easily changed
            for g := 1 to ColCount do
            begin
              DNT_sql_col := aItems[g].sql_col;
              if not Progress.isRunning then
                break;

              if (DNT_sql_col = DNT_AccessedTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 1); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_ModifiedTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 2); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_ExpiryTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 3); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_CreationTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 4); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_SyncTime) then
                variant_Array[g] := CarveDate(hdr_start_of_regex_match, 5); // Date 1 is the closest to regex match
              if (DNT_sql_col = DNT_AccessCount) then
                variant_Array[g] := CarveAccessCount(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_Filename) then
                variant_Array[g] := CarveFileName(hdr_last_matched_char - 8); // -8 is back from http
              if (DNT_sql_col = DNT_FileOffset) then
                variant_Array[g] := SetFileOffset(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_Flags) then
                variant_Array[g] := CarveFlags(hdr_start_of_regex_match);
              if (DNT_sql_col = DNT_RecoveryType) then
                variant_Array[g] := SetRecoveryType(Item.fi_Recovery_Type);
              if (DNT_sql_col = DNT_Reference) then
                variant_Array[g] := CarveReference(hdr_last_matched_char);
              if (DNT_sql_col = DNT_Url) then
                variant_Array[g] := CarveURL(hdr_last_matched_char);
              if (DNT_sql_col = DNT_UrlHash) then
                variant_Array[g] := CarveURLHash(hdr_start_of_regex_match);
            end;
            AddToModule;
            if VERBOSE then
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

            // Restore the position and find the next match
            HeaderReader.Position := curpos;
            HeaderRegex.FindNext(hdr_start_of_regex_match, hdr_count_of_matched_bytes);
          end;
        except
          MessageUser('Error processing ' + aArtifactEntry.EntryName);
        end;
      end;
    end;
  end;

  procedure nodefrog(const ProcessID: string; aNode: TPropertyNode; anint: integer);
  var
    idx: integer;
    theNodeList: TObjectList;
    pad_str: string;
    bl_found: boolean;
    bl_logging: boolean;
    NumSeconds: integer;
    SafariDateTime: TDateTime;
  begin
    bl_logging := False;
    pad_str := (StringOfChar(' ', anint * 3));
    if assigned(aNode) and Progress.isRunning then
    begin
      theNodeList := aNode.PropChildList;
      if assigned(theNodeList) then
      begin
        if bl_logging then
          Progress.Log(pad_str + aNode.PropName); // only print the header if the node has children
        bl_found := False;

        for idx := 0 to theNodeList.Count - 1 do
        begin
          if not Progress.isRunning then break;
          aNode := TPropertyNode(theNodeList.items[idx]);

          for g := 1 to ColCount do
          begin
            DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));

            // Safari BRW_SAFARI_HISTORYPLIST URL
            if
            (Item.fi_Process_ID = 'BRW_SAFARI_HISTORYPLIST') and
            (UpperCase(aNode.PropName) = '') and
            (DNT_sql_col = 'URL') and
            RegexMatch(aNode.PropDisplayValue, 'http', false) then
            begin
              variant_Array[g] := aNode.PropDisplayValue;
            end
            else

            if (UpperCase(aNode.PropName) = UpperCase(DNT_sql_col)) then
            begin
              // ---------------------------------------------------------------
              // Safari
              if ((UpperCase(aNode.PropName) = '') and (aItems[g].col_type = ftString) and (DNT_sql_col = 'URL') and (aItems[g].convert_as = 'SafariURL')) then // Safari History Plist URL - Blank Property name
              begin
                if aNode.PropName = 'date_added' then // noslz
                begin
                  try
                    variant_Array[g] := Int64ToDateTime_ConvertAs(StrToInt64(aNode.PropDisplayValue), 'DTChrome'); // KVL - Force the conversion type.
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error converting string to int.');
                  end;
                end;
              end
              else

              // Safari BRW_SAFARI_HISTORYPLIST Last Visit Date
              if (Item.fi_Process_ID = 'BRW_SAFARI_HISTORYPLIST') and (UpperCase(aNode.PropName) = 'LASTVISITEDDATE') then // noslz
              begin
                try
                  variant_Array[g] := MacAbsoluteTimeAsDoubleToDateTime(StrToFloat(aNode.PropDisplayValue));
                except
                  Progress.Log(ATRY_EXCEPT_STR + 'Error converting float string to datetime.');
                end;
              end
              else

              // Only DT values
              if (aItems[g].col_type = ftDateTime) then
              begin
                if (ProcessID = 'DNT_SAFARI_HISTORY_XML') and (UpperCase(aNode.PropName) = 'LASTVISITEDDATE') and (aItems[g].convert_as = 'SafariDT') then
                begin
                  NumSeconds := trunc(StrToFloatWithDecimalPoint(aNode.PropDisplayValue));
                  SafariDateTime := EncodeDateTime(2001, 1, 1, 0, 0, 0, 0);
                  SafariDateTime := IncSecond(SafariDateTime, NumSeconds);
                  variant_Array[g] := SafariDateTime;
                end
                else

                // -------------------------------------------------------------
                // Chrome or Edge
                if ((ProcessID = 'BRW_BRAVE_BOOKMARKS') or (ProcessID = 'BRW_CHROME_BOOKMARKS') or (ProcessID = 'BRW_EDGE_BOOKMARKS') or (ProcessID = 'BRW_OPERA_BOOKMARKS')) and (aItems[g].convert_as = 'DTChrome') then
                begin
                  try
                    variant_Array[g] := Int64ToDateTime_ConvertAs(StrToInt64(aNode.PropDisplayValue), aItems[g].convert_as);
                  except
                  end;
                end
                else if variant_Array[g] = null then // Once there is a value in the array, do not use one from deeper
                  variant_Array[g] := aNode.PropDisplayValue;
              end
              else

              // ---------------------------------------------------------------
              // The rest...
              begin
                if variant_Array[g] = null then // Once there is a value in the array, do not use one from deeper
                begin
                  variant_Array[g] := aNode.PropDisplayValue;
                end;
              end;

              if bl_logging then
                Progress.Log(RPad(pad_str + aNode.PropName, RPAD_VALUE) + aNode.PropDisplayValue);
              bl_found := True;
            end;
          end;
          nodefrog(ProcessID, aNode, anint + 1);
        end;
      end;
      if bl_found then
      begin
        AddToModule; // Also nulls the array
        if bl_logging then
          Progress.Log(StringOfChar('-', CHAR_LENGTH));
        NullTheArray;
      end;
    end;
  end;

  procedure nodefrog2(const ProcessID: string; aNode: TPropertyNode; anint: integer); // Use for list of key then string, key then string
  var
    idx, k: integer;
    theNodeList: TObjectList;
    pad_str: string;
    nextNode: TPropertyNode;
    nodefollow: boolean;
  begin
    pad_str := (StringOfChar(' ', anint * 3));
    if assigned(aNode) and Progress.isRunning then
    begin
      theNodeList := aNode.PropChildList;
      if assigned(theNodeList) then
      begin
        nodefollow := True;
        for idx := 0 to theNodeList.Count - 1 do
        begin
          aNode := TPropertyNode(theNodeList.items[idx]);
          for k := 1 to ColCount do
          begin
            DNT_sql_col := copy(aItems[k].sql_col, 5, Length(aItems[k].sql_col));
            if aNode.PropName = 'key' then // noslz
            begin
              if UpperCase(aNode.PropDisplayValue) = (UpperCase(DNT_sql_col)) then
              begin
                if idx + 1 <= theNodeList.Count then
                begin
                  nextNode := TPropertyNode(theNodeList.items[idx + 1]);
                  Progress.Log(RPad(pad_str + aNode.PropDisplayValue, RPAD_VALUE) + nextNode.PropDisplayValue);
                  variant_Array[k] := nextNode.PropDisplayValue;
                  nodefollow := False;
                end;
              end;
            end;
          end;
          if nodefollow then
            nodefrog2(ProcessID, aNode, anint + 1);
        end;
        Progress.Log(StringOfChar('-', CHAR_LENGTH));
        AddToModule;
        NullTheArray;
      end;
    end;
  end;

// Start of Do Process =========================================================
var
  process_proceed_bl: boolean;
  temp_flt: Double;
  temp_int64: int64;
  temp_process_counter: integer;

begin
  if gArtifactsDataStore = nil then
    Exit;

  Item := @gArr[ref_num];
  temp_process_counter := 0;
  ColCount := LengthArrayTABLE(aItems);

  if (UpperCase(Item.fi_Process_As) = 'POSTPROCESS') then
  begin
    Exit;
  end;

  if (gArr_ValidatedFiles_TList[ref_num].Count > 0) then
  begin
    process_proceed_bl := True;
    if process_proceed_bl then
    begin
      SetLength(variant_Array, ColCount + 1);
      ADDList := TList.Create;
      newEntryReader := TEntryReader.Create;
      try
        Progress.Max := gArr_ValidatedFiles_TList[ref_num].Count;
        Progress.DisplayMessageNow := 'Process' + SPACE + PROGRAM_NAME + ' - ' + Item.fi_Name_OS + RUNNING;
        Progress.CurrentPosition := 1;

        // Regex Setup ---------------------------------------------------------
        CarvedEntryReader := TEntryReader.Create;
        FooterProgress := TPAC.Create;
        FooterProgress.Start;
        FooterReader := TEntryReader.Create;
        FooterRegEx := TRegEx.Create;
        FooterRegEx.CaseSensitive := True;
        FooterRegEx.Progress := FooterProgress;
        FooterRegEx.SearchTerm := Item.fi_Carve_Footer;
        HeaderReader := TEntryReader.Create;
        HeaderRegex := TRegEx.Create;
        HeaderRegex.CaseSensitive := True;
        HeaderRegex.Progress := Progress;
        HeaderRegex.SearchTerm := Item.fi_Carve_Header;

        if HeaderRegex.LastError <> 0 then
        begin
          Progress.Log('HeaderRegex Error: ' + IntToStr(HeaderRegex.LastError));
          aArtifactEntry := nil;
          Exit;
        end;

        if FooterRegEx.LastError <> 0 then
        begin
          Progress.Log(RPad('!!!!!!! FOOTER REGEX ERROR !!!!!!!:', RPAD_VALUE) + IntToStr(FooterRegEx.LastError));
          aArtifactEntry := nil;
          Exit;
        end;

        TotalFiles := gArr_ValidatedFiles_TList[ref_num].Count;

        begin
          // Loop Validated Files ------------------------------------------------
          for i := 0 to gArr_ValidatedFiles_TList[ref_num].Count - 1 do { addList is freed a the end of this loop }
          begin
            if not Progress.isRunning then
              break;

            Progress.IncCurrentprogress;
            temp_process_counter := temp_process_counter + 1;
            Display_Name_str := GetFullName(Item);
            Progress.DisplayMessageNow := 'Processing' + SPACE + Display_Name_str + ' (' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gArr_ValidatedFiles_TList[ref_num].Count) + ')' + RUNNING;
            aArtifactEntry := TEntry(gArr_ValidatedFiles_TList[ref_num].items[i]);

            if assigned(aArtifactEntry) and newEntryReader.OpenData(aArtifactEntry) and (newEntryReader.Size > 0) and Progress.isRunning then
            begin
              if BL_USE_FLAGS then aArtifactEntry.Flags := aArtifactEntry.Flags + [Flag8]; // Gray Flag = Process Routine

              //Progress.Log(StringOfChar('F', 100) + SPACE + Item.fi_Process_ID + ' >>> ' + Item.fi_Process_As + ' >>> ' + Item.fi_ESE_ContainerName);

              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_ESE_TABLES)                   then ProcessAsESE;
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_ESE_CARVE_CONTENT)            then ProcessAsESECarveContent;
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_ESE_CARVE_COOKIES)            then ProcessAsESECarveCookies;
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_ESE_CARVE_HISTORY_VISITED)    then ProcessAsESECarveHistoryVisited;
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_ESE_CARVE_HISTORY_REF_MARKER) then ProcessAsESECarveHistoryRefMarker;

              // =================================================================
              // PROCESS AS: Nodes - .PropName, .PropDisplayValue, .PropValue, .PropDataType
              // =================================================================
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_PLIST) then
              begin
                aPropertyTree := nil;
                try
                  if ProcessMetadataProperties(aArtifactEntry, aPropertyTree, newEntryReader) and assigned(aPropertyTree) then
                  begin
                    // Set Root Property (Level 0)
                    aRootProperty := aPropertyTree;
                    // Progress.Log(format('%-21s %-26s %-10s %-10s %-20s ',['Number of root child nodes:',IntToStr(aRootProperty.PropChildList.Count),'Bates:',IntToStr(aArtifactEntry.ID),aArtifactEntry.EntryName]));
                    if assigned(aRootProperty) and assigned(aRootProperty.PropChildList) and (aRootProperty.PropChildList.Count >= 1) then
                    begin
                      if not Progress.isRunning then
                        break;

                      // Chrome Bookmarks
                      if (UpperCase(Item.fi_Process_ID) = 'BRW_CHROME_BOOKMARKS') and RegexMatch(aArtifactEntry.FullPathName, 'Chrome', False) then // noslz
                      begin
                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                        if assigned(Node1) then
                        begin
                          NumberOfNodes := CountNodes(Node1);
                          nodefrog(Item.fi_Process_ID, Node1, 0);
                        end;
                      end;

                      // Edge Bookmarks
                      if (UpperCase(Item.fi_Process_ID) = 'BRW_EDGE_BOOKMARKS') and RegexMatch(aArtifactEntry.FullPathName, 'Edge', False) then // noslz
                      begin
                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                        if assigned(Node1) then
                        begin
                          NumberOfNodes := CountNodes(Node1);
                          nodefrog(Item.fi_Process_ID, Node1, 0);
                        end;
                      end;

                      // Opera Bookmarks
                      if (UpperCase(Item.fi_Process_ID) = 'DNT_OPERA_BOOKMARKS') and RegexMatch(aArtifactEntry.FullPathName, 'Opera', False) then // noslz
                      begin
                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                        if assigned(Node1) then
                        begin
                          NumberOfNodes := CountNodes(Node1);
                          nodefrog(Item.fi_Process_ID, Node1, 0);
                        end;
                      end;

                      if (RegexMatch(Item.fi_Process_ID, 'BRW_IE_49_', True)) and ((UpperCase(Item.fi_Name_Program_Type) = 'URL (COOKIE)') or (UpperCase(Item.fi_Name_Program_Type) = 'URL (DAILY)') or
                        (UpperCase(Item.fi_Name_Program_Type) = 'URL (HISTORY)') or (UpperCase(Item.fi_Name_Program_Type) = 'URL (NORMAL)')) then
                      begin
                        for x := 0 to aRootProperty.PropChildList.Count - 1 do
                        begin
                          if not Progress.isRunning then
                            break;
                          Node1 := TPropertyNode(aRootProperty.PropChildList.items[x]);
                          if assigned(Node1) then
                          begin
                            NodeList1 := Node1.PropChildList;
                            NumberOfNodes := NodeList1.Count;
                            // Cycle Through Child List - Level 2
                            for y := 0 to NumberOfNodes - 1 do
                            begin
                              Node2 := TPropertyNode(NodeList1.items[y]);
                              if assigned(Node2) then
                              begin
                                for g := 1 to ColCount do
                                begin
                                  if not Progress.isRunning then
                                    break;
                                  // Add to the Array
                                  DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
                                  if UpperCase(Node2.PropName) = UpperCase(DNT_sql_col) then
                                  begin
                                    variant_Array[g] := Node2.PropDisplayValue;
                                  end;
                                end;
                              end;
                            end;
                          end;
                          AddToModule;
                        end;
                      end;

                      if (UpperCase(Item.fi_Process_ID) = 'BRW_SAFARI_BOOKMARKSPLIST') then
                      begin
                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                        if assigned(Node1) then
                        begin
                          NumberOfNodes := CountNodes(Node1);
                          NullTheArray;
                          nodefrog(Item.fi_Process_ID, Node1, 0);
                        end;
                      end;

                      if (UpperCase(Item.fi_Process_ID) = 'BRW_SAFARI_COOKIES') then
                      begin
                        if aRootProperty.PropName = RS_DNT_SAFARI_COOKIE_DATA then
                        begin
                          for x := 0 to aRootProperty.PropChildList.Count - 1 do
                          begin
                            if not Progress.isRunning then
                              break;
                            Node1 := TPropertyNode(aRootProperty.PropChildList.items[x]);
                            // Get the 2nd Node
                            if assigned(Node1) and Progress.isRunning then
                            begin
                              NodeList1 := Node1.PropChildList;
                              if assigned(NodeList1) then
                              begin
                                for y := 0 to NodeList1.Count - 1 do
                                begin
                                  Node2 := TPropertyNode(NodeList1.items[y]);
                                  // Get the 3rd Node
                                  if assigned(Node2) and Progress.isRunning then
                                  begin
                                    NodeList2 := Node2.PropChildList;
                                    if assigned(NodeList2) then
                                    begin
                                      for z := 0 to NodeList2.Count - 1 do
                                      begin
                                        Node3 := TPropertyNode(NodeList2.items[z]);
                                        // Populate
                                        if assigned(Node3) and Progress.isRunning then
                                        begin
                                        if not Progress.isRunning then
                                        break;
                                        if assigned(Node3) then
                                        begin
                                        for g := 1 to ColCount do
                                        begin
                                        if not Progress.isRunning then
                                        break;
                                        DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col)); // Add to the Array
                                        if UpperCase(Node3.PropName) = UpperCase(DNT_sql_col) then
                                        variant_Array[g] := Node3.PropDisplayValue;
                                        end;
                                        end;
                                        end;
                                      end;
                                    end;
                                  end;
                                end;
                              end;
                            end;
                            AddToModule;
                          end;
                        end;
                      end;

                      if (UpperCase(Item.fi_Process_ID) = 'BRW_SAFARI_DOWNLOADSXML') then
                      begin
                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                        if assigned(Node1) then
                        begin
                          NumberOfNodes := CountNodes(Node1);
                          NullTheArray;
                          nodefrog2(Item.fi_Process_ID, Node1, 0);
                        end;
                      end;

                      if (UpperCase(Item.fi_Process_ID) = 'BRW_SAFARI_DOWNLOADSPLIST') then
                      begin
                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                        if assigned(Node1) then
                        begin
                          NumberOfNodes := CountNodes(Node1);
                          NullTheArray;
                          nodefrog(Item.fi_Process_ID, Node1, 0);
                        end;
                      end;

                      if (UpperCase(Item.fi_Process_ID) = 'BRW_SAFARI_HISTORYPLIST') then
                      begin
                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                        if assigned(Node1) then
                        begin
                          NumberOfNodes := CountNodes(Node1);
                          NullTheArray;
                          nodefrog(Item.fi_Process_ID, Node1, 0);
                        end;
                      end;

                      if (UpperCase(Item.fi_Process_ID) = 'BRW_SAFARI_RECENTSEARCHESPLIST') then
                      begin
                        TestNode1 := aRootProperty.GetFirstNodeByName(Item.fi_NodeByName); // noslz
                        if assigned(TestNode1) then
                        begin
                          Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                          if assigned(Node1) then
                          begin
                            NumberOfNodes := CountNodes(Node1);
                            nodefrog(Item.fi_Process_ID, Node1, 0);
                          end;
                        end;
                      end; { Safari Recent Searches PList }

                      if (UpperCase(Item.fi_Process_ID) = 'BRW_SAFARI_RECENTSEARCHESXML') then
                      begin
                        TestNode1 := aRootProperty.GetFirstNodeByName(RS_DNT_KEY);
                        TestNode2 := aRootProperty.GetFirstNodeByName('plist');
                        if assigned(TestNode2) and assigned(TestNode1) and (UpperCase(TestNode1.PropValue) = 'RECENTSEARCHES') then // This is the test for whether it is a Safari Recent Searches
                        begin
                          Temp_NodeList := TList.Create;
                          try
                            aRootProperty.FindNodesByName(RS_DNT_STRING, Temp_NodeList);
                            for x := 0 to Temp_NodeList.Count - 1 do
                            begin
                              Node1 := TPropertyNode(Temp_NodeList.items[x]);
                              if assigned(Node1) and (Node1.PropName = RS_DNT_STRING) then
                                for g := 1 to ColCount do
                                  variant_Array[g] := Node1.PropValue;
                              AddToModule;
                            end;
                          finally
                            if assigned(Temp_NodeList) then
                              FreeAndNil(Temp_NodeList);
                          end;
                        end;
                      end; { Safari Recent Searches XML }

                      if (UpperCase(Item.fi_Process_ID) = 'BRW_SAFARI_TOPSITESPLIST') then
                      begin
                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                        if assigned(Node1) then
                        begin
                          NumberOfNodes := CountNodes(Node1);
                          NullTheArray;
                          nodefrog(Item.fi_Process_ID, Node1, 0);
                        end;
                      end;
                    end
                    else
                      Progress.Log((format('%-39s %-10s %-10s', ['Could not open data:', '-', 'Bates: ' + IntToStr(aArtifactEntry.ID) + ' ' + aArtifactEntry.EntryName])));
                  end;
                finally
                  if assigned(aPropertyTree) then
                    FreeAndNil(aPropertyTree);
                end;
              end;

              // =================================================================
              // PROCESS AS: SQL
              // =================================================================
              if RegexMatch(Item.fi_Signature_Parent, RS_SIG_SQLITE, False) then
              begin
                newWALReader := GetWALReader(gFileSystemDataStore, aArtifactEntry);
                newJOURNALReader := GetJOURNALReader(gFileSystemDataStore, aArtifactEntry);
                try
                  mydb := TSQLite3Database.Create;
                  try
                    mydb.OpenStream(newEntryReader, newJOURNALReader, newWALReader);

                    if Test_SQL_Tables(ref_num) then
                    begin
                      records_read_int := 0;

                      // ***************************************************************************************************************************
                      sqlselect := TSQLite3Statement.Create(mydb, (Item.fi_SQLStatement));
                      //Progress.Log(StringOfChar('~', CHAR_LENGTH));
                      //Progress.Log(Item.fi_SQLStatement);
                      //Progress.Log(StringOfChar('~', CHAR_LENGTH));
                      // ***************************************************************************************************************************

                      while sqlselect.Step = SQLITE_ROW do
                      begin
                        if not Progress.isRunning then
                          break;
                        records_read_int := records_read_int + 1;

                        // Progress for large files
                        if sql_row_count > 15000 then
                          Progress.DisplayMessages := 'Processing' + SPACE + Display_Name_str + ' (' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gArr_ValidatedFiles_TList[ref_num].Count) + ')' + SPACE + IntToStr(records_read_int) + '/' + IntToStr(sql_row_count) + RUNNING;

                        // Read the values from the SQL tables -----------------
                        for g := 1 to ColCount do
                        begin
                          if not Progress.isRunning then
                            break;
                          DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));

                          // Chrome SyncData - Specifics Blob Bytes
                          url_str := '';
                          title_str := '';
                          fav_str := '';
                          if (DNT_sql_col = 'Specifics') and (aItems[g].convert_as = 'SyncData') then
                          begin
                            variant_Array[g] := '';
                            test_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                            sSize := Length(test_bytes);

                            // Chrome SyncData Standard URL
                            if (Length(test_bytes) >= 10) and (test_bytes[0] = $C2) and (test_bytes[1] = $88) and (test_bytes[2] = $10) then
                            begin
                              startpos := 6;
                              if (test_bytes[3] >= $80) then
                              begin
                                Inc(startpos);
                                if (test_bytes[4] >= $01) then
                                  Inc(startpos);
                              end;
                              if (test_bytes[startpos - 2] = $0A) then
                              begin
                                Size := test_bytes[startpos - 1];
                                for b := 1 to Size do
                                  url_str := url_str + chr(test_bytes[startpos + b - 1]);

                                startpos := startpos + Size;
                                if (test_bytes[startpos] = $1A) then
                                begin
                                  startpos := startpos + 2;
                                  Size := test_bytes[startpos - 1];
                                  for b := 1 to Size do
                                    title_str := title_str + chr(test_bytes[startpos + b - 1]);
                                end;
                              end;
                              if url_str <> '' then
                                variant_Array[g] := url_str + ' ' + title_str;
                            end
                            else
                            begin // Chrome SyncData FavIcon
                              if (Length(test_bytes) >= 10) and (test_bytes[0] = $9A) and (test_bytes[1] = $F0) and (test_bytes[2] = $58) then
                              begin
                                startpos := 6;
                                if (test_bytes[3] >= $80) then
                                  Inc(startpos);
                                if (test_bytes[startpos - 2] = $0A) then
                                begin
                                  Size := test_bytes[startpos - 1];
                                  for b := 1 to Size do
                                    fav_str := fav_str + chr(test_bytes[startpos + b - 1]);
                                end;
                              end;
                              if fav_str <> '' then
                                variant_Array[g] := fav_str;
                            end;
                          end

                          else if aItems[g].read_as = ftString then
                            variant_Array[g] := ColumnValueByNameAsText(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftinteger) then
                            variant_Array[g] := ColumnValueByNameAsInt(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftString) then
                            variant_Array[g] := ColumnValueByNameAsInt(sqlselect, DNT_sql_col)

                          //else if (aItems[g].read_as = ftLargeInt) and (aItems[g].col_type = ftLargeInt) then
                            //variant_Array[g] := ColumnValueByNameAsint64(sqlselect, DNT_sql_col)

                          else if (aItems[g].col_type = ftDateTime) then
                          begin
                            if (ColumnValueByNameAsint64(sqlselect, aItems[g].field_name) > 0) then
                            begin
                              try
                                if aItems[g].read_as = ftFloat then
                                begin
                                  temp_flt := ColumnValueByNameAsint64(sqlselect, aItems[g].field_name);
                                  variant_Array[g] := GHFloatToDateTime(temp_flt, aItems[g].convert_as);
                                end
                                else
                                begin
                                  temp_int64 := ColumnValueByNameAsint64(sqlselect, aItems[g].field_name);
                                  variant_Array[g] := Int64ToDateTime_ConvertAs(temp_int64, aItems[g].convert_as);
                                end;
                              except
                                on e: exception do
                                  Progress.Log(e.message);
                              end;
                            end;
                          end;

                          // Add the table name and row location
                          if DNT_sql_col = 'SQLLOCATION' then
                            variant_Array[g] := 'Table: ' + lowercase(Item.fi_SQLPrimary_Tablestr) + ' (row ' + format('%.*d', [4, records_read_int]) + ')'; // noslz

                          // SPECIAL CONVERSIONS FOLLOW ==========================

  ////                          // Convert Cache.db url_cache_response from string to date time
  //                          if (UpperCase(DNT_sql_col) = 'TIME_STAMP') and (aItems[g].read_as = ftString) and (col_DF[g].FieldType = ftDateTime) then
  //                          begin
  ////                            try
  ////                              Progress.Log('kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk');
  ////                              //variant_Array[g] := VarToDateTime(variant_Array[g]);
  ////                            except
  ////                              MessageUser('Could not covert Cache.db url_cahce_response from string to datetime.'); // noslz
  ////                            end;
  //                          end;

                          if aItems[g].convert_as = 'DOWNLOADSTATE' then
                          begin
                            // Source: https://github.com/sans-dfir/sift-files/blob/master/scripts/hindsight.py
                            if variant_Array[g] = '0' then
                              variant_Array[g] := 'In Progress';
                            if variant_Array[g] = '1' then
                              variant_Array[g] := 'Complete';
                            if variant_Array[g] = '2' then
                              variant_Array[g] := 'Canceled';
                            if variant_Array[g] = '2' then
                              variant_Array[g] := 'Interrupted';
                          end;

                          if aItems[g].convert_as = 'OPENEDSTATE' then
                          begin
                            if variant_Array[g] = '1' then
                              variant_Array[g] := 'Yes';
                            if variant_Array[g] = '0' then
                              variant_Array[g] := 'No';
                          end;

                          if aItems[g].convert_as = 'INTERRUPTREASON' then
                          begin
                            // Source: https://github.com/sans-dfir/sift-files/blob/master/scripts/hindsight.py - April 2017
                            if variant_Array[g] = '0' then variant_Array[g] := '';
                            if variant_Array[g] = '1' then variant_Array[g] := 'File Error';
                            if variant_Array[g] = '2' then variant_Array[g] := 'Access Denied';
                            if variant_Array[g] = '3' then variant_Array[g] := 'Disk Full';
                            if variant_Array[g] = '5' then variant_Array[g] := 'Path Too Long';
                            if variant_Array[g] = '6' then variant_Array[g] := 'File Too Large';
                            if variant_Array[g] = '7' then variant_Array[g] := 'Virus';
                            if variant_Array[g] = '10' then variant_Array[g] := 'Temporary Problem';
                            if variant_Array[g] = '11' then variant_Array[g] := 'Blocked';
                            if variant_Array[g] = '12' then variant_Array[g] := 'Security Check Failed';
                            if variant_Array[g] = '13' then variant_Array[g] := 'Resume Error';
                            if variant_Array[g] = '20' then variant_Array[g] := 'Network Error';
                            if variant_Array[g] = '21' then variant_Array[g] := 'Operation Timed Out';
                            if variant_Array[g] = '22' then variant_Array[g] := 'Connection Lost';
                            if variant_Array[g] = '23' then variant_Array[g] := 'Server Down';
                            if variant_Array[g] = '30' then variant_Array[g] := 'Server Error';
                            if variant_Array[g] = '31' then variant_Array[g] := 'Range Request Error';
                            if variant_Array[g] = '32' then variant_Array[g] := 'Server Precondition Error';
                            if variant_Array[g] = '33' then variant_Array[g] := 'Unable to get file';
                            if variant_Array[g] = '40' then variant_Array[g] := 'Canceled';
                            if variant_Array[g] = '41' then variant_Array[g] := 'Browser Shutdown';
                            if variant_Array[g] = '50' then variant_Array[g] := 'Browser Crashed';
                          end;

                          if aItems[g].convert_as = 'TRANSITIONTYPE' then
                          // Run a conversion in this loop if required -----------
                          begin
                            // Source: https://github.com/sans-dfir/sift-files/blob/master/scripts/hindsight.py - April 2017
                            //developer.chrome.com/extensions/history
                            if variant_Array[g] = '0' then variant_Array[g] := 'link';
                            if variant_Array[g] = '1' then variant_Array[g] := 'typed';
                            if variant_Array[g] = '2' then variant_Array[g] := 'auto bookmark';
                            if variant_Array[g] = '3' then variant_Array[g] := 'auto subframe';
                            if variant_Array[g] = '4' then variant_Array[g] := 'manual subframe';
                            if variant_Array[g] = '5' then variant_Array[g] := 'generated';
                            if variant_Array[g] = '6' then variant_Array[g] := 'start page';
                            if variant_Array[g] = '7' then variant_Array[g] := 'form submit';
                            if variant_Array[g] = '8' then variant_Array[g] := 'reload';
                            if variant_Array[g] = '9' then variant_Array[g] := 'keyword';
                            if variant_Array[g] = '10' then variant_Array[g] := 'keyword generated';
                          end;

                          if aItems[g].convert_as = 'DANGERTYPE' then
                          begin
                            if variant_Array[g] = '0' then variant_Array[g] := 'Not Dangerous';
                            if variant_Array[g] = '1' then variant_Array[g] := 'Dangerous';
                            if variant_Array[g] = '2' then variant_Array[g] := 'Dangerous URL';
                            if variant_Array[g] = '3' then variant_Array[g] := 'Dangerous Content';
                            if variant_Array[g] = '4' then variant_Array[g] := 'Content May Be Malicious';
                            if variant_Array[g] = '5' then variant_Array[g] := 'Uncommon Content';
                            if variant_Array[g] = '6' then variant_Array[g] := 'Dangerous But User Validated';
                            if variant_Array[g] = '7' then variant_Array[g] := 'Dangerous Host';
                            if variant_Array[g] = '8' then variant_Array[g] := 'Potentially Unwanted';
                          end;

                        end;
                        AddToModule;
                      end;
                      if assigned(sqlselect) then sqlselect.free;
                      Progress.Log(RPad(HYPHEN + 'Bates:' + SPACE + IntToStr(aArtifactEntry.ID) + HYPHEN + copy(aArtifactEntry.EntryName, 1, 40), RPAD_VALUE) + format('%-5s %-5s', [IntToStr(records_read_int), '(SQL)']));
                    end;
                  finally
                    FreeAndNil(mydb);
                    if assigned(newWALReader) then
                      FreeAndNil(newWALReader);
                    if assigned(newJOURNALReader) then
                      FreeAndNil(newJOURNALReader);
                  end;
                except
                  on e: exception do
                  begin
                    Progress.Log(ATRY_EXCEPT_STR + RPad('WARNING:', RPAD_VALUE) + 'ERROR DOING SQL QUERY!');
                    Progress.Log(e.message);
                  end;
                end;
              end;

              // ===============================================================
              // PROCESS AS: REGEX CARVE
              // ===============================================================
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_CARVE) then
              begin
                if HeaderReader.OpenData(aArtifactEntry) then // Open the entry to search
                begin
                  try
                    Progress.Initialize(aArtifactEntry.PhysicalSize, 'Carving ' + IntToStr(i + 1) + ' of ' + IntToStr(TotalFiles) + ' files: ' + aArtifactEntry.EntryName);
                    h_startpos := 0;
                    curpos := 0;
                    HeaderReader.Position := 0;
                    HeaderRegex.Stream := HeaderReader;

                    // Opera
                    if Item.fi_Name_Program = RS_OPERA then
                    begin
                      gh_int := 0;
                      if HeaderReader.OpenData(aArtifactEntry) and FooterReader.OpenData(aArtifactEntry) then // Open the entry to search
                        try
                          Progress.Initialize(aArtifactEntry.PhysicalSize, 'Searching ' + IntToStr(i + 1) + ' of ' + IntToStr(TotalFiles) + ' files: ' + aArtifactEntry.EntryName);
                          h_startpos := 0;
                          HeaderReader.Position := 0;
                          HeaderRegex.Stream := HeaderReader;
                          FooterRegEx.Stream := FooterReader;
                          HeaderRegex.Find(h_offset, h_count); // Find the first match, h_offset returned is relative to start pos
                          while h_offset <> -1 do // h_offset returned as -1 means no hit
                          begin
                            FooterRegEx.Stream.Position := h_offset + h_count; // Advance so it does not match itself
                            end_pos := h_offset + h_count + 10000; // Limit looking for the footer to our max size
                            if end_pos >= HeaderRegex.Stream.Size then
                              end_pos := HeaderRegex.Stream.Size - 1; // Don't go past the end!
                            FooterRegEx.Stream.Size := end_pos;
                            FooterRegEx.Find(f_offset, f_count);
                            if (f_offset <> -1) and (f_offset + f_count >= 1) then // Found a footer and the size was at least our minimum
                            begin
                              CarvedEntry := TEntry.Create;
                              try
                                CarvedData := TByteInfo.Create; // ByteInfo is data described as a list of byte runs, usually just one run
                                CarvedData.ParentInfo := aArtifactEntry.DataInfo; // Point to the data of the file
                                CarvedData.RunLstAddPair(h_offset, f_offset + f_count); // Adds the block of data
                                CarvedEntry.DataInfo := CarvedData;
                                CarvedEntry.LogicalSize := CarvedData.Datasize;
                                CarvedEntry.PhysicalSize := CarvedData.Datasize;
                                CarvedEntryReader := TEntryReader.Create;
                                try
                                  if CarvedEntryReader.OpenData(CarvedEntry) then
                                  begin
                                    CarvedEntryReader.Position := 0;
                                    carved_str := '';
                                    carved_str := CarvedEntryReader.AsPrintableChar(CarvedEntryReader.Size);
                                    // Progress.Log(RPad(inttostr(gh_int) + SPACE + 'Carved Str:', 20) + carved_str);
                                    Inc(gh_int);
                                    if carved_str <> '' then
                                    begin
                                      variant_Array[1] := carved_str;
                                      AddToModule;
                                    end;
                                  end;
                                finally
                                  CarvedEntryReader.free;
                                end;
                              finally
                                CarvedEntry.free;
                              end;
                            end;
                            HeaderRegex.FindNext(h_offset, h_count); // Find each subsequent header match for the current file
                          end;
                        except
                          Progress.Log(ATRY_EXCEPT_STR + 'Error processing ' + aArtifactEntry.EntryName);
                        end;
                    end
                    else
                      // Chrome
                      if (Item.fi_Name_Program = RS_CHROME) or (Item.fi_Name_Program = RS_CHROMIUM) then
                      begin
                        // Chrome Cache ------------------------------------------
                        // Chrome Cache entry - http://forensicswiki.org/wiki/Chrome_Disk_Cache_Format
                        // Back from URL   // The first cache entry is at offset 8192. The cache entry (struct EntryStore) is 256 bytes in size and consists of:
                        // 96              // 0  4 - Hash - The hash of the key
                        // 92              // 4  4 - Next entry cache address - The next entry with the same hash or bucket
                        // 88              // 8  4 - Rankings node cache address
                        // 84              // 12 4 - Reuse count - Value that indicates how often this entry was (re-)used
                        // 80              // 16 4 - Refetch count - Value that indicates how often this entry was (re-)fetched (or downloaded)
                        // 76              // 20 4 - Cache entry state
                        // 72              // 24 8 - Creation datetime
                        // 64              // 32 4 - Key data size (key length)
                        // 60              // 36 4 - Long key data cache address. The value is 0 if no long key data is present
                        // 56              // 40 4 - Data Size Headers
                        // 52              // 44 4 - Data Size Payload
                        // 48              // 48 4 - Data Size
                        // 44              // 52 4 - Data Size
                        // 40              // 56 4 - Cache Address - HTTP Headers
                        // 36              // 60 4 - Cache Address -
                        // 32              // 64 4 - Cache Address
                        // 28              // 68 4 - Cache Address
                        // 24              // 72 4           Cache entry flags
                        // 20              // 76 4 x 4 = 16  Padding
                        // 4               // 92 4           Self hash. The hash of the first 92 bytes of the cache entry is this used as a checksum?
                        // 0               // 96 160         Key data - Contains an UTF-8 encoded string with an end-of-string character with the original URL
                        // --------------------------------------------------------------------------

                        Log_ChromeCache_bl := False;
                        Progress.Log(RPad('Searching ' + IntToStr(i + 1) + ' of ' + IntToStr(TotalFiles) + ' files: ', RPAD_VALUE) + aArtifactEntry.EntryName);
                        HeaderRegex.Find(h_offset, h_count);

                        // Find the first match, h_offset returned is relative to start pos
                        while h_offset <> -1 do // h_offset returned as -1 means no hit
                        begin
                          variant_Array[1] := null;
                          variant_Array[2] := null;
                          curpos := HeaderReader.Position;

                          if Log_ChromeCache_bl then
                            Progress.Log(RPad('Searching ' + IntToStr(i + 1) + ' of ' + IntToStr(TotalFiles) + ' files: ', RPAD_VALUE) + aArtifactEntry.EntryName);
                          if Log_ChromeCache_bl then
                            Progress.Log(RPad('Found http at offset: ', RPAD_VALUE) + IntToStr(h_offset));

                          if (h_startpos + h_offset - 72) < 0 then // 72 is DateTime + URL length
                          begin
                            Progress.Log('Error: Start position cannot be before file start.');
                            break;
                          end
                          else
                          begin
                            // Get the URL datetime ------------------------------
                            HeaderReader.Position := (h_startpos + h_offset - 72);
                            url_dt := 0.0;
                            if HeaderReader.read(url_unixdt, 8) = 8 then
                              try
                                url_dt := int64ToDateTime(url_unixdt);
                                dt_str := DateTimeToStr(url_dt);
                              except
                                Progress.Log(ATRY_EXCEPT_STR);
                              end;
                            if Log_ChromeCache_bl then
                              Progress.Log(RPad('Date Time: ', RPAD_VALUE) + dt_str);

                            // Get DataSize --------------------------------------
                            HeaderReader.Position := (h_startpos + h_offset - 52);
                            data_size := 0;
                            if Log_ChromeCache_bl then
                              Progress.Log(RPad('Data Size: ', RPAD_VALUE) + IntToStr(data_size));

                            // Get Reuse Count -----------------------------------
                            HeaderReader.Position := (h_startpos + h_offset - 84);
                            reuse_count := 0;
                            if Log_ChromeCache_bl then
                              Progress.Log(RPad('Reuse Count: ', RPAD_VALUE) + IntToStr(reuse_count));

                            // Get Refetch Count ---------------------------------
                            HeaderReader.Position := (h_startpos + h_offset - 80);
                            refetch_count := 0;
                            if Log_ChromeCache_bl then
                              Progress.Log(RPad('Refetch Count: ', RPAD_VALUE) + IntToStr(refetch_count));

                            // Get the url length --------------------------------
                            HeaderReader.Position := (h_startpos + h_offset - 64);
                            // URL length is stored in 4 bytes 64 bytes before the URL
                            url_str := '';
                            if HeaderReader.read(url_len, 4) = 4 then
                            begin
                              if (url_len >= 10) and (url_len <= 1024) then
                              begin
                                if Log_ChromeCache_bl then
                                  Progress.Log(RPad('URL Length: ', RPAD_VALUE) + IntToStr(url_len));

                                HeaderReader.Position := (h_startpos + h_offset); // Go back to the start of the URL
                                setlength(ByteArray, url_len);

                                if HeaderReader.read(ByteArray[0], url_len) = url_len then
                                  for b := 0 to url_len - 1 do
                                    url_str := url_str + chr(ByteArray[b]);
                                if Log_ChromeCache_bl then
                                  Progress.Log(RPad('URL: ', RPAD_VALUE) + url_str);
                              end;
                            end
                            else
                              Progress.Log('Failed to read stream at position: ' + IntToStr(HeaderReader.Position));

                            // Add results to Array
                            if url_str <> '' then
                            begin
                              if (trunc(url_dt) >= 37622) and (trunc(url_dt) < 80000) then // 2003 to 2118 approx.
                              begin
                                variant_Array[1] := url_dt;
                                variant_Array[2] := url_str;
                                variant_Array[3] := h_offset;
                                variant_Array[4] := url_len;
                                variant_Array[5] := data_size;
                                variant_Array[6] := reuse_count;
                                variant_Array[7] := refetch_count;
                                AddToModule;
                              end;
                            end;

                          end;
                          HeaderReader.Position := curpos;
                          HeaderRegex.FindNext(h_offset, h_count); // Find each subsequent header match
                          if Log_ChromeCache_bl then
                            Progress.Log(StringOfChar('-', CHAR_LENGTH));
                        end;
                      end;
                  except
                    MessageUser('Error processing ' + aArtifactEntry.EntryName);
                  end; { Regex Carve }
                end;
              end;

            end;

            if Log_ChromeCache_bl then
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

            Progress.IncCurrentprogress;
          end;

        end;

        // Add to the gArtifactsDataStore
        if assigned(ADDList) and Progress.isRunning then
        begin
          if (CmdLine.Params.Indexof(TRIAGE) = -1) then
            gArtifactsDataStore.Add(ADDList);
          Progress.Log(RPad('Total Artifacts:', RPAD_VALUE) + IntToStr(ADDList.Count));

          // Export L01 files where artifacts are found (controlled by Artifact_Utils.pas)
          if (gArr_ValidatedFiles_TList[ref_num].Count > 0) and (ADDList.Count > 0) then
            ExportToL01(gArr_ValidatedFiles_TList[ref_num], Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type);

          ADDList.Clear;
        end;

        Progress.Log(StringOfChar('-', CHAR_LENGTH));
      finally
        records_read_int := 0;
        FreeAndNil(FooterProgress);
        FreeAndNil(FooterReader);
        FreeAndNil(FooterRegEx);
        FreeAndNil(HeaderReader);
        FreeAndNil(HeaderRegex);
        FreeAndNil(ADDList);
        FreeAndNil(newEntryReader);
      end;
    end;

  end;

//  else
//  begin
//    Progress.Log(RPad('Process List #' + IntToStr(ref_num) + SPACE + '(' + IntToStr(gArr_ValidatedFiles_TList[ref_num].Count) + '):', RPAD_VALUE) + gArr[ref_num].fi_Name_Program + SPACE + gArr[ref_num].fi_Name_Program_Type + RUNNING);
//    //Progress.Log(RPad('', RPAD_VALUE) + 'No files to process.');
//    Progress.Log(StringOfChar('-', CHAR_LENGTH));
//  end;

end;

// ==========================================================================================================================================================
// Start of Script
// ==========================================================================================================================================================
const
  CREATE_GLOB_SEARCH_BL = False;
  SEARCHING_FOR_ARTIFACT_FILES_STR = 'Searching for Artifact files';

var
  AboutToProcess_StringList: TStringList;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  FolderEntry: TEntry;
  AllFoundList_count: integer;
  AllFoundListUnique: TUniqueListOfEntries;
  DeleteFolder_display_str: string;
  DeleteFolder_TList: TList;
  Enum: TEntryEnumerator;
  ExistingFolders_TList: TList;
  FieldItunesDomain: TDataStoreField;
  FieldItunesName: TDataStoreField;
  FindEntries_StringList: TStringList;
  FoundList, AllFoundList: TList;
  GlobSearch_StringList: TStringList;
  i: integer;
  Item: TSQL_FileSearch;
  iTunes_Domain_str: string;
  iTunes_Name_str: string;
  Process_ID_str: string;
  ResultInt: integer;

{$IF DEFINED (ISFEXGUI)}
  Display_StringList: TStringList;
  MyScriptForm: TScriptForm;
{$IFEND}

procedure LogExistingFolders(aExistingFolders_TList: TList; aDeleteFolder_TList: TList); // Compare About to Process Folders with Existing Artifact Module Folders
const
  LOG_BL = False;
var
  aFolderEntry: TEntry;
  idx, t: integer;
begin
  if LOG_BL then Progress.Log('Existing Folders:');
  for idx := 0 to aExistingFolders_TList.Count - 1 do
  begin
    if not Progress.isRunning then break;
    aFolderEntry := (TEntry(aExistingFolders_TList[idx]));
    if LOG_BL then
      Progress.Log(HYPHEN + aFolderEntry.FullPathName);
    for t := 0 to AboutToProcess_StringList.Count - 1 do
    begin
      if aFolderEntry.FullPathName = AboutToProcess_StringList[t] then
        aDeleteFolder_TList.Add(aFolderEntry);
    end;
  end;
  if LOG_BL then
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
end;

procedure Post_Process(artifacts_TList: TList);
const
  REGEX_BING_QUERY      = 'https?\:\/\/((www)\.)?bing\.com\/.*search\?';
  REGEX_CLOUD_URL       = 'dropbox\.|\.box\.|onedrive\.|drive\.google|pcloud\.|idrive\.';
  REGEX_DATING_URL      = '(adultfriendfinder|ashleymadison|badoo|bumble|eharmony|elitesingles|facebookdating|happon|grinder|match|okcupid|plentyoffish|tinder|zoosk)\.com';
  REGEX_DEEPFAKE_URL    = 'https?\:\/\/((www)\.)?deepswap|faceswap|fotor|pica-ai|pixble|reface|swapper|vidnoz';
  REGEX_FACEBOOK_URL    = 'https?\:\/\/((www)\.)?facebook\.com';
  REGEX_GOOGLE_QUERY    = '(https|http):\/\/(www\.|)google\.[a-z]+[\.a-z]+\/search\?';
  REGEX_MAPS_QUERY      = 'google\.com\/maps(\?=|\/place\/|\/dir\/)';
  REGEX_SHIPPING_URL    = 'dtdc|(aramex|bluedart|dbschenker|deutschepost|dhl|fedex|jdlogistics|Purolator|royalmail|ups|yrcfreight|zto)\.com';
  REGEX_YOUTUBE_QUERY   = 'https?\:\/\/((www)\.)?youtube\.com\/results\?search_query=';

var
  aItems:               TSQL_Table_array;
  col_DF:               TDataStoreFieldArray;
  ColCount:             integer;
  DNT_sql_col:          string;
  full_url_str:         string;
  i:                    integer;
  Item:                 TSQL_FileSearch;
  NEntry:               TArtifactItem;
  query_str:            string;
  ref_num:              integer;

  ADDList:              TList;
  Bing_TList:           TList;
  Cloud_TList:          TList;
  Dating_TList:         TList;
  DeepFake_TList:       TList;
  Facebook_TList:       TList;
  GoogleMaps_TList:     TList;
  GoogleQuery_TList:    TList;
  Shipping_TList:       TList;
  YouTube_TList:        TList;

  procedure AddPostProcessData(aTList: TList; ProcID_str: string; name_str1: string; name_str2: string; icon1: integer; icon2: integer);
  var
    aProgFldr: TArtifactConnectEntry;
    Entry: TEntry;
    i,g: integer;
    Item: TSQL_FileSearch;
    url_str: string;
  begin
    if aTList.Count > 0 then
    begin
      aProgFldr := AddArtifactConnect(TEntry(gArtConnect_CatFldr), name_str1, name_str2, '', icon1, icon2);
      aProgFldr.Status := aProgFldr.Status + [dstUserCreated];

      // Look for the fi_Process_ID in the Array to determin the ref_num
      ref_num := -1;
      for i := 0 to Length(gArr) - 1 do
      begin
        Item := gArr[i];
        if Item.fi_Process_ID = ProcID_str then
        begin
          ref_num := i;
          break;
        end;
      end;

      // Read then setup the columns
      aItems := Browser_Columns.GetTable(ProcID_str);
      ColCount := LengthArrayTABLE(Browser_Columns.GetTable(ProcID_str));
      if ref_num = -1 then
        Exit
      else
        SetUpColumnforFolder(ref_num, aProgFldr, col_DF, ColCount, Browser_Columns.GetTable(ProcID_str));

      // Process each entry in the list
      for i := 0 to aTList.Count - 1 do
      begin
        // Create a new artifact entry
        NEntry := TArtifactItem.Create;
        NEntry.SourceEntry := Entry;
        NEntry.Parent := aProgFldr;
        NEntry.PhysicalSize := 0;
        NEntry.LogicalSize := 0;

        // Get the details from the TList entry
        Entry := TEntry(aTList[i]);

        // Get URL String
        if assigned(gcol_url) then
          url_str := gcol_url.AsString(Entry);

        // Get Query String
        query_str := '';
        if RegexMatch(url_str, REGEX_MAPS_QUERY,    False) then query_str := trim(PerlMatch('(?<=\/place\/|\/dir\/)(.*)(?=\/@|\/data)', url_str)) else
        if RegexMatch(url_str, REGEX_YOUTUBE_QUERY, False) then query_str := trim(PerlMatch('(?<=search_query\=)(.*)(?=&|$)', url_str))
        else
          query_str := trim(PerlMatch('(?<=q=)(.*)(?=&|$)', url_str));
        query_str := trim(CleanString(query_str));

        // Fill the columns
        for g := 1 to ColCount do
        begin
          //Progress.Log(DNT_sql_col);
          if not Progress.isRunning then break;
          DNT_sql_col := aItems[g].sql_col;
          if DNT_sql_col = 'DNT_ENTRY_NAME' then col_DF[g].AsString[NEntry] := gcol_source_file.AsString(Entry);
          if DNT_sql_col = 'DNT_ENTRY_PATH' then col_DF[g].AsString[NEntry] := gcol_source_path.AsString(Entry);
          if DNT_sql_col = 'DNT_QUERY'      then col_DF[g].AsString[NEntry] := query_str;
          if DNT_sql_col = 'DNT_URL'        then col_DF[g].AsString[NEntry] := url_str;
        end;
        ADDList.Add(NEntry);
      end;
      // Add to the Artifacts module
      gArtifactsDataStore.Add(ADDList);
      ADDList.Clear;
    end;
  end;

var
  art_Entry: TEntry;
begin
  if not assigned(gArtConnect_CatFldr) then
  begin
    Progress.Log('Not assigned an artifacts category folder.' + SPACE + TSWT); // noslz
    Exit;
  end;

  if assigned(artifacts_TList) and (artifacts_TList.Count > 0) then
  begin
    Progress.Log(RPad('Artifacts Module Count:', RPAD_VALUE) + IntToStr(artifacts_TList.Count));

    ADDList := TList.Create;
    Bing_TList := TList.Create;
    Cloud_TList := TList.Create;
    Dating_TList := TList.Create;
    DeepFake_TList := TList.Create;
    Facebook_TList := TList.Create;
    GoogleMaps_TList := TList.Create;
    GoogleQuery_TList := TList.Create;
    Shipping_TList := TList.Create;
    Youtube_TList := TList.Create;

    try
      Progress.DisplayMessageNow := 'Collecting summary URL information' + RUNNING;
      Progress.Initialize(artifacts_TList.Count, 'Collecting summary URL information' + RUNNING);
      for i := 0 to artifacts_TList.Count - 1 do
      begin
        if not Progress.isRunning then break;
        Progress.incCurrentProgress;
        art_Entry := TEntry(artifacts_TList[i]);

        if assigned(gcol_url) then
          full_url_str := gcol_url.AsString(art_Entry);

        if (full_url_str = null) or (full_url_str = '') then
          Continue;

        if RegexMatch(full_url_str, REGEX_BING_QUERY, False)    then Bing_TList.Add(art_Entry);
        if RegexMatch(full_url_str, REGEX_CLOUD_URL, False)     then Cloud_TList.Add(art_Entry);
        if RegexMatch(full_url_str, REGEX_DATING_URL, False)    then Dating_TList.Add(art_Entry);
        if RegexMatch(full_url_str, REGEX_DEEPFAKE_URL, False)  then Deepfake_TList.Add(art_Entry);
        if RegexMatch(full_url_str, REGEX_FACEBOOK_URL, False)  then Facebook_TList.Add(art_Entry);
        if RegexMatch(full_url_str, REGEX_GOOGLE_QUERY, False)  then GoogleQuery_TList.Add(art_Entry);
        if RegexMatch(full_url_str, REGEX_MAPS_QUERY, False)    then GoogleMaps_TList.Add(art_Entry);
        if RegexMatch(full_url_str, REGEX_SHIPPING_URL, False)  then Shipping_TList.Add(art_Entry);
        if RegexMatch(full_url_str, REGEX_YOUTUBE_QUERY, False) then YouTube_TList.Add(art_Entry);
      end;

      Progress.Log(RPad('REGEX_BING_QUERY:',    RPAD_VALUE) + IntToStr(Bing_TList.Count));
      Progress.Log(RPad('REGEX_CLOUD_URL:',     RPAD_VALUE) + IntToStr(Cloud_TList.Count));
      Progress.Log(RPad('REGEX_DATING_URL:',    RPAD_VALUE) + IntToStr(Dating_TList.Count));
      Progress.Log(RPad('REGEX_DEEPFAKE_URL:',  RPAD_VALUE) + IntToStr(DeepFake_TList.Count));
      Progress.Log(RPad('REGEX_FACEBOOK_URL:',  RPAD_VALUE) + IntToStr(Facebook_TList.Count));
      Progress.Log(RPad('REGEX_GOOGLE_QUERY:',  RPAD_VALUE) + IntToStr(GoogleQuery_TList.Count));
      Progress.Log(RPad('REGEX_MAPS_QUERY:',    RPAD_VALUE) + IntToStr(GoogleMaps_TList.Count));
      Progress.Log(RPad('REGEX_SHIPPING_URL:',  RPAD_VALUE) + IntToStr(Shipping_TList.Count));
      Progress.Log(RPad('REGEX_YOUTUBE_QUERY:', RPAD_VALUE) + IntToStr(YouTube_TList.Count));

      AddPostProcessData(Bing_TList,        'BRW_SUMMARY_QUERY',    '@ Bing',     'Query',    1142, 997);
      AddPostProcessData(Cloud_TList,       'BRW_SUMMARY_CLOUD',    '@ Cloud',    'Services', 1231, 997);
      AddPostProcessData(Dating_TList,      'BRW_SUMMARY_DATING',   '@ Dating',   'URL',      463,  997);
      AddPostProcessData(Deepfake_TList,    'BRW_SUMMARY_DEEPFAKE', '@ Deepfake', 'URL',      1188, 997);
      AddPostProcessData(Facebook_TList,    'BRW_SUMMARY_FACEBOOK', '@ Facebook', 'URL',      1018, 997);
      AddPostProcessData(GoogleMaps_TList,  'BRW_SUMMARY_QUERY',    '@ Google',   'Maps',     1139, 997);
      AddPostProcessData(GoogleQuery_TList, 'BRW_SUMMARY_QUERY',    '@ Google',   'Query',    1139, 997);
      AddPostProcessData(Shipping_TList,    'BRW_SUMMARY_SHIPPING', '@ Shipping', 'URL',      1213, 997);
      AddPostProcessData(YouTube_TList,     'BRW_SUMMARY_QUERY',    '@ YouTube',  'Query',    1141, 997);

    finally
      ADDList.free;
      Bing_TList.free;
      Cloud_TList.free;
      Dating_TList.free;
      DeepFake_TList.free;
      Facebook_TList.free;
      GoogleMaps_TList.free;
      GoogleQuery_TList.free;
      Shipping_TList.free;
      Youtube_TList.free;
    end;
  end;
end;

// Start of script -------------------------------------------------------------
var
  anEntry: TEntry;
  anint: integer;
  param_str: string;
  progress_program_str: string;
  ref_num: integer;
  regex_search_str: string;
  n, r, s: integer;
  PostProcess_TList: TList;
  temp_int: integer;
  test_param_int: integer;
  temp_process_counter: integer;

begin
  Read_SQLite_DB;

  SetLength(gArr_ValidatedFiles_TList, gNumberOfSearchItems);
  SetLength(gArtConnect_ProgFldr, gNumberOfSearchItems);

  Progress.Log(SCRIPT_NAME + ' started' + RUNNING);
  if (CmdLine.Params.Indexof(TRIAGE) > -1) then
    Progress.DisplayTitle := 'Triage' + HYPHEN + 'File System' + HYPHEN + CATEGORY_NAME
  else
    Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME;
  Progress.LogType := ltVerbose; // ltOff, ltVerbose, ltDebug, ltTechnical

  if not StartingChecks then
  begin
    Progress.DisplayMessageNow := ('Processing complete.');
    Exit;
  end;

  // Parameters Received -------------------------------------------------------
  param_str := '';
  test_param_int := 0;
  gParameter_Num_StringList := TStringList.Create;
  try
//    if CmdLine.ParamCount = 0 then
//    begin
//      MessageUser('No processing parameters received.');
//      Exit;
//    end
//    else
    begin
      Progress.Log(RPad('Parameters Received:', RPAD_VALUE) + IntToStr(CmdLine.ParamCount));
      for n := 0 to CmdLine.ParamCount - 1 do
      begin
        if not Progress.isRunning then
          break;

        param_str := param_str + '"' + CmdLine.Params[n] + '"' + ' ';
        // Validate Parameters
        if (RegexMatch(CmdLine.Params[n], '\d{1,2}$', False)) then
        begin
          try
            test_param_int := StrToInt(CmdLine.Params[n]);
            if (test_param_int <= -1) or (test_param_int > Length(gArr) - 1) then
            begin
              MessageUser('Invalid parameter received: ' + (CmdLine.Params[n]) + CR + ('Maximum is: ' + IntToStr(Length(gArr) - 1) + DCR + SCRIPT_NAME + ' will terminate.'));
              Exit;
            end;
            gParameter_Num_StringList.Add(CmdLine.Params[n]);
            Item := gArr[test_param_int];
            Progress.Log(RPad(HYPHEN + 'param_str ' + IntToStr(n) + ' = ' + CmdLine.Params[n], RPAD_VALUE) + format('%-10s %-10s %-25s %-12s', ['Ref#: ' + CmdLine.Params[n], CATEGORY_NAME,
              Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, Item.fi_Name_OS]));
          except
            begin
              Progress.Log(ATRY_EXCEPT_STR + 'Error validating parameter. ' + SCRIPT_NAME + ' will terminate.');
              Exit;
            end;
          end;
        end;
      end;
      Trim(param_str);
    end;
    Progress.Log(RPad('param_str:', RPAD_VALUE) + param_str); // noslz
    Progress.Log(StringOfChar('-', CHAR_LENGTH));

    // Progress Bar Text
    progress_program_str := '';
    if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      progress_program_str := PROGRAM_NAME;
    end;

    // Create GlobSearch -------------------------------------------------------
    if CREATE_GLOB_SEARCH_BL then
    begin
      GlobSearch_StringList := TStringList.Create;
      GlobSearch_StringList.Sorted := True;
      GlobSearch_StringList.Duplicates := dupIgnore;
      try
        for n := 0 to Length(gArr) - 1 do
          Create_Global_Search(n, gArr[n], GlobSearch_StringList);
        Progress.Log(GlobSearch_StringList.Text);
      finally
        GlobSearch_StringList.free;
      end;
      Exit;
    end;

    for n := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      if (CmdLine.Params.Indexof(IntToStr(n)) > -1) then
      begin
        Item := gArr[n];
        progress_program_str := 'Ref#:' + SPACE + param_str + SPACE + Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type;
        break;
      end;
    end;

    if progress_program_str = '' then
      progress_program_str := 'Other';

    gArtifactsDataStore := GetDataStore(DATASTORE_ARTIFACTS);
    if not assigned(gArtifactsDataStore) then
    begin
      Progress.Log(DATASTORE_ARTIFACTS + ' module not located.' + DCR + TSWT);
      Exit;
    end;

    try
      gFileSystemDataStore := GetDataStore(DATASTORE_FILESYSTEM);
      if not assigned(gFileSystemDataStore) then
      begin
        Progress.Log(DATASTORE_FILESYSTEM + ' module not located.' + DCR + TSWT);
        Exit;
      end;

      try
        FieldItunesDomain := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_DOMAIN);
        FieldItunesName := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_NAME);

        // Create TLists For Valid Files
        for n := 0 to Length(gArr) - 1 do
        begin
          if not Progress.isRunning then
            break;
          gArr_ValidatedFiles_TList[n] := TList.Create;
        end;

        try
          AboutToProcess_StringList := TStringList.Create;
          try
            if (CmdLine.Params.Indexof('MASTER') = -1) then
            begin
              if (CmdLine.Params.Indexof('NOSHOW') = -1) then
              begin
                // PROCESSALL - Create the AboutToProcess_StringList
                if (CmdLine.Params.Indexof(PROCESSALL) > -1) or (CmdLine.ParamCount = 0) then
                begin
                  for n := 0 to Length(gArr) - 1 do
                  begin
                    if not Progress.isRunning then break;
                    Item := gArr[n];
                    AboutToProcess_StringList.Add(CATEGORY_NAME + BS + GetFullName(@gArr[n]));
                    Progress.Log(format('%-4s %-59s %-30s', ['#' + IntToStr(n), Item.fi_Process_ID, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type]));
                  end;
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end
                else

                // PROCESS INDIVIDUAL - Create the AboutToProcess_StringList
                begin
                  if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
                  begin
                    for n := 0 to gParameter_Num_StringList.Count - 1 do
                    begin
                      if not Progress.isRunning then break;
                      temp_int := StrToInt(gParameter_Num_StringList[n]);
                      Item := gArr[temp_int];
                      AboutToProcess_StringList.Add(CATEGORY_NAME + BS + GetFullName(@gArr[temp_int]));
                    end;
                  end;
                end;

                if CmdLine.ParamCount = 0 then
                begin
                  Progress.Log('No parameters received. Use a number or "PROCESSALL".');
                  Exit;
                end;

                // Show the form
                ResultInt := 1; // Continue AboutToProcess

{$IF DEFINED (ISFEXGUI)}
                Display_StringList := TStringList.Create;
                Display_StringList.Sorted := True;
                Display_StringList.Duplicates := dupIgnore;
                try
                  for i := 0 to AboutToProcess_StringList.Count - 1 do
                  begin
                    Display_StringList.Add(ExtractFileName(AboutToProcess_StringList[i]));
                  end;

                  if assigned(AboutToProcess_StringList) and (AboutToProcess_StringList.Count > 30) then
                  begin
                    MyScriptForm := TScriptForm.Create;
                    try
                      MyScriptForm.SetCaption(SCRIPT_DESCRIPTION);
                      MyScriptForm.SetText(Display_StringList.Text);
                      ResultInt := idCancel;
                      if MyScriptForm.ShowModal then
                        ResultInt := idOk
                    finally
                      FreeAndNil(MyScriptForm);
                    end;
                  end
                  else
                  begin
                    ResultInt := MessageBox('Extract Artifacts?', 'Extract Artifacts' + HYPHEN + PROGRAM_NAME, (MB_OKCANCEL or MB_ICONQUESTION or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                  end;
                finally
                  Display_StringList.free;
                end;
{$IFEND}
              end;

              if ResultInt > 1 then
              begin
                Progress.Log(CANCELED_BY_USER);
                Progress.DisplayMessageNow := CANCELED_BY_USER;
                Exit;
              end;

            end;

            // Deal with Existing Artifact Folders
            ExistingFolders_TList := TList.Create;
            DeleteFolder_TList := TList.Create;
            try
              if gArtifactsDataStore.Count > 1 then
              begin
                anEntry := gArtifactsDataStore.First;
                while assigned(anEntry) and Progress.isRunning do
                begin
                  if anEntry.isDirectory then
                    ExistingFolders_TList.Add(anEntry);
                  anEntry := gArtifactsDataStore.Next;
                end;
                gArtifactsDataStore.Close;
              end;

              LogExistingFolders(ExistingFolders_TList, DeleteFolder_TList);

              // Create the delete folder TList and display string
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                Progress.Log('Replacing folders:');
                for s := 0 to DeleteFolder_TList.Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  FolderEntry := TEntry(DeleteFolder_TList[s]);
                  DeleteFolder_display_str := DeleteFolder_display_str + #13#10 + FolderEntry.FullPathName;
                  Progress.Log(HYPHEN + FolderEntry.FullPathName);
                end;
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
              end;

              // Message Box - When artifact folder is already present ---------------------
{$IF DEFINED (ISFEXGUI)}
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                if (CmdLine.Params.Indexof('MASTER') = -1) and (CmdLine.Params.Indexof('NOSHOW') = -1) then
                begin

                  if (CmdLine.Params.Indexof('NOSHOW') = -1) then
                  begin
                    if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
                    begin
                      ResultInt := MessageBox('Artifacts have already been processed:' + #13#10 + DeleteFolder_display_str + DCR + 'Replace the existing Artifacts?', 'Extract Artifacts' + HYPHEN + PROGRAM_NAME,
                        (MB_OKCANCEL or MB_ICONWARNING or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                    end;
                  end
                  else
                    ResultInt := idOk;
                  case ResultInt of
                    idOk:
                      begin
                        try
                          gArtifactsDataStore.Remove(DeleteFolder_TList);
                        except
                          MessageBox('ERROR: There was an error deleting existing artifacts.' + #13#10 + 'Save then reopen your case.', SCRIPT_NAME, (MB_OK or MB_ICONINFORMATION or MB_SETFOREGROUND or MB_TOPMOST));
                        end;
                      end;
                    idCancel:
                      begin
                        Progress.Log(CANCELED_BY_USER);
                        Progress.DisplayMessageNow := CANCELED_BY_USER;
                        Exit;
                      end;
                  end;
                end;
              end;
{$IFEND}

            finally
              ExistingFolders_TList.free;
              DeleteFolder_TList.free;
            end;
          finally
            AboutToProcess_StringList.free;
          end;

          // Find iTunes Backups and run Signature Analysis
          if (CmdLine.Params.Indexof('NOITUNES') = -1) then
            Itunes_Backup_Signature_Analysis;

          // Create the RegEx Search String
          regex_search_str := '';
          begin
            for n := 0 to Length(gArr) - 1 do
            begin
              if not Progress.isRunning then
                break;
              Item := gArr[n];
              if (CmdLine.Params.Indexof(IntToStr(n)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
              begin
                if Item.fi_Regex_Search <> ''      then regex_search_str := regex_search_str + '|' + Item.fi_Regex_Search;
                if Item.fi_Rgx_Itun_Bkup_Dmn <> '' then regex_search_str := regex_search_str + '|' + Item.fi_Rgx_Itun_Bkup_Dmn;
                if Item.fi_Rgx_Itun_Bkup_Nme <> '' then regex_search_str := regex_search_str + '|' + Item.fi_Rgx_Itun_Bkup_Nme;
              end;
            end;
          end;
          if (regex_search_str <> '') and (regex_search_str[1] = '|') then
            Delete(regex_search_str, 1, 1);

          AllFoundList := TList.Create;
          try
            AllFoundListUnique := TUniqueListOfEntries.Create;
            FoundList := TList.Create;
            FindEntries_StringList := TStringList.Create;
            FindEntries_StringList.Sorted := True;
            FindEntries_StringList.Duplicates := dupIgnore;
            try
              if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
              begin
                Progress.Log('Find files by path (PROCESSALL)' + RUNNING);
                for i := low(gArr) to high(gArr) do
                begin
                  if (gArr[i].fi_Glob1_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob1_Search);
                  if (gArr[i].fi_Glob2_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob2_Search);
                  if (gArr[i].fi_Glob3_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob3_Search);
                end;
                // Add Overflow Glob Search Items
                FindEntries_StringList.Add('**\1d6740792a2b845f4c1e6220c43906d7f0afe8ab.mddata'); // noslz
              end
              else
              begin
                Progress.Log('Find files by path (Individual)' + RUNNING);
                for i := 0 to gParameter_Num_StringList.Count - 1 do
                begin
                  anint := StrToInt(gParameter_Num_StringList[i]);
                  if (gArr[anint].fi_Glob1_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob1_Search);
                  if (gArr[anint].fi_Glob2_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob2_Search);
                  if (gArr[anint].fi_Glob3_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob3_Search);
                end;
              end;

              // Find the files by path and add to AllFoundListUnique
              Progress.Initialize(FindEntries_StringList.Count, STR_FILES_BY_PATH + RUNNING);
              gtick_foundlist_i64 := GetTickCount;
              for i := 0 to FindEntries_StringList.Count - 1 do
              begin
                if not Progress.isRunning then
                  break;
                try
                  Find_Entries_By_Path(gFileSystemDataStore, FindEntries_StringList[i], FoundList, AllFoundListUnique);
                except
                  Progress.Log(RPad(ATRY_EXCEPT_STR, RPAD_VALUE) + 'Find_Entries_By_Path');
                end;
                Progress.IncCurrentprogress;
              end;

              Progress.Log(RPad(STR_FILES_BY_PATH + SPACE + '(Unique)' + COLON, RPAD_VALUE) + IntToStr(AllFoundListUnique.Count));
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

              // Add > 40 char files with no extension to Unique List (possible iTunes backup files)
              Add40CharFiles(AllFoundListUnique);
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

              // Move the AllFoundListUnique list into a TList
              if assigned(AllFoundListUnique) and (AllFoundListUnique.Count > 0) then
              begin
                Enum := AllFoundListUnique.GetEnumerator;
                while Enum.MoveNext do
                begin
                  anEntry := Enum.Current;
                  AllFoundList.Add(anEntry);
                end;
              end;

              // Now work with the TList from now on
              if assigned(AllFoundList) and (AllFoundList.Count > 0) then
              begin
                Progress.Log(RPad('Unique Files by path:', RPAD_VALUE) + IntToStr(AllFoundListUnique.Count));
                Progress.Log(StringOfChar('-', CHAR_LENGTH));

                // Add flags
                if BL_USE_FLAGS then
                begin
                  Progress.Log('Adding flags' + RUNNING);
                  for i := 0 to AllFoundList.Count - 1 do
                  begin
                    if not Progress.isRunning then
                      break;
                    anEntry := TEntry(AllFoundList[i]);
                    anEntry.Flags := anEntry.Flags + [Flag7];
                  end;
                  Progress.Log('Finished adding flags' + RUNNING);
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Determine signature
                if assigned(AllFoundList) and (AllFoundList.Count > 0) then
                  SignatureAnalysis(AllFoundList);

                gtick_foundlist_str := (RPad(STR_FILES_BY_PATH + COLON, RPAD_VALUE) + CalcTimeTaken(gtick_foundlist_i64));
              end;

            finally
              FoundList.free;
              AllFoundListUnique.free;
              FindEntries_StringList.free;
            end;

            if assigned(AllFoundList) and (AllFoundList.Count > 0) then
            begin
              Progress.Log(SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              Progress.Log(format(GFORMAT_STR, ['', 'Action', 'Ref#', 'Bates', 'Signature', 'Filename (trunc)', 'Reason'])); // noslz
              Progress.Log(format(GFORMAT_STR, ['', '------', '----', '-----', '---------', '----------------', '------'])); // noslz

              AllFoundList_count := AllFoundList.Count;
              Progress.Initialize(AllFoundList_count, SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              for i := 0 to AllFoundList.Count - 1 do
              begin
                Progress.DisplayMessages := SEARCHING_FOR_ARTIFACT_FILES_STR + SPACE + '(' + IntToStr(i) + ' of ' + IntToStr(AllFoundList_count) + ')' + RUNNING;
                if not Progress.isRunning then
                  break;
                anEntry := TEntry(AllFoundList[i]);

                if (i mod 10000 = 0) and (i > 0) then
                begin
                  Progress.Log('Processing: ' + IntToStr(i) + ' of ' + IntToStr(AllFoundList.Count) + RUNNING);
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Set the iTunes Domain String
                iTunes_Domain_str := '';
                if assigned(FieldItunesDomain) then
                begin
                  try
                    iTunes_Domain_str := FieldItunesDomain.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Domain string');
                  end;
                end;

                // Set the iTunes Name String
                iTunes_Name_str := '';
                if assigned(FieldItunesName) then
                begin
                  try
                    iTunes_Name_str := FieldItunesName.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Name string');
                  end;
                end;

                // Run the match
                aDeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
                if
                (anEntry.Extension <> 'db-shm') and
                (anEntry.Extension <> 'db-wal') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite WAL') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite SHM') then
                begin
                  if RegexMatch(anEntry.EntryName, regex_search_str, False) or
                  RegexMatch(anEntry.FullPathName, regex_search_str, False) or
                  FileSubSignatureMatch(anEntry) or
                  RegexMatch(iTunes_Domain_str, regex_search_str, False) or
                  RegexMatch(iTunes_Name_str, regex_search_str, False) then
                  begin

                    // Sub Signature Match
                    if FileSubSignatureMatch(anEntry) then
                    begin
                      if BL_USE_FLAGS then anEntry.Flags := anEntry.Flags + [Flag2]; // Blue Flag
                      DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                    end
                    else

                    // Regex Name Match
                    begin
                      if ((not anEntry.isDirectory) or (anEntry.isDevice)) and (anEntry.LogicalSize > 0) and (anEntry.PhysicalSize > 0) then
                      begin
                        if FileNameRegexSearch(anEntry, iTunes_Domain_str, iTunes_Name_str, regex_search_str) then
                        begin
                          if BL_USE_FLAGS then anEntry.Flags := anEntry.Flags + [Flag1]; // Red Flag
                          DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                        end;
                      end;
                    end;

                  end;
                  Progress.IncCurrentprogress;
                end;
              end;
            end;
          finally
            AllFoundList.free;
          end;

          // Check to see if files were found
          if (TotalValidatedFileCountInTLists = 0) then
          begin
            Progress.Log('No ' + PROGRAM_NAME + SPACE + 'files were found.');
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Not found';
          end
          else
          begin
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
            Progress.Log(RPad('Total Validated Files:', RPAD_VALUE) + IntToStr(TotalValidatedFileCountInTLists));
            Progress.Log(StringOfChar('=', CHAR_LENGTH + 80));
          end;

          // Display the content of the TLists for further processing
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.Log('Lists available to process' + RUNNING);
            for n := 0 to Length(gArr) - 1 do
            begin
              if not Progress.isRunning then
                break;
              Item := gArr[n];
              if gArr_ValidatedFiles_TList[n].Count > 0 then
              begin
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
                Progress.Log(RPad('TList ' + IntToStr(n) + ': ' + Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type + HYPHEN + Item.fi_Name_OS, RPAD_VALUE) + IntToStr(gArr_ValidatedFiles_TList[n].Count));
                for r := 0 to gArr_ValidatedFiles_TList[n].Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  anEntry := (TEntry(TList(gArr_ValidatedFiles_TList[n]).items[r]));
                  if not(Item.fi_Process_As = 'POSTPROCESS') then
                    Progress.Log(RPad(' Bates: ' + IntToStr(anEntry.ID), RPAD_VALUE) + anEntry.EntryName + SPACE + Item.fi_Process_As);
                end;
              end;
            end;
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
          end;

          // *** CREATE GUI ***
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Found';

            Progress.Log(RPad('Process All Lists' + RUNNING, RPAD_VALUE) + IntToStr(High(gArr) + 1)); // noslz
            Progress.Log(StringOfChar('=', CHAR_LENGTH));

            // *** DO PROCESS ***
            temp_process_counter := 0;
            Progress.CurrentPosition := 0;
            Progress.Max := TotalValidatedFileCountInTLists;
            gtick_doprocess_i64 := GetTickCount;

            // *****************************************************************
            for n := 0 to Length(gArr) - 1 do
            begin
              ref_num := n;
              if (TestForDoProcess(ref_num)) then
              begin
                Item := gArr[n];
                Process_ID_str := Item.fi_Process_ID;
                DoProcess(gArtConnect_ProgFldr[ref_num], ref_num, Browser_Columns.GetTable(Process_ID_str));
              end;
            end;
            // *****************************************************************

            gtick_doprocess_str := (RPad('DoProcess:', RPAD_VALUE) + CalcTimeTaken(gtick_doprocess_i64));

          end;

          if not Progress.isRunning then
          begin
            Progress.Log(CANCELED_BY_USER);
            Progress.DisplayMessageNow := CANCELED_BY_USER;
          end;

        finally
          for n := 0 to Length(gArr) - 1 do
            FreeAndNil(gArr_ValidatedFiles_TList[n]);
        end;

      finally
        if assigned(gFileSystemDataStore) then
          FreeAndNil(gFileSystemDataStore);
      end;

    finally
      if assigned(gArtifactsDataStore) then
        FreeAndNil(gArtifactsDataStore);
    end;

  finally
    if assigned(gParameter_Num_StringList) then
      FreeAndNil(gParameter_Num_StringList);
  end;

  if Progress.isRunning then
  begin
    Progress.Log(gtick_foundlist_str);
    Progress.Log(gtick_doprocess_str);

    // Post Process
    gArtifactsDataStore := GetDataStore(DATASTORE_ARTIFACTS);
    if not assigned(gArtifactsDataStore) then
    begin
      Progress.Log(DATASTORE_ARTIFACTS + ' module not located.' + DCR + TSWT);
      Exit;
    end;
    try
      PostProcess_TList := gArtifactsDataStore.GetEntireList;
      try
        Post_Process(PostProcess_TList);
      finally
        PostProcess_TList.free;
      end;
    finally
      gArtifactsDataStore.free;
    end;
  end;

  Progress.Log(StringOfChar('-', CHAR_LENGTH));

  Progress.Log(SCRIPT_NAME + ' finished.');
  Progress.DisplayMessageNow := ('Processing complete.');

end.
